import AppContext from '@/context/AppContext'
import { signOut, useSession } from 'next-auth/react';
import Image from 'next/image';
import { useRouter } from 'next/router';
import React, { useContext, useState } from 'react'
import { IoMdAdd } from "react-icons/io";
import PageLoading from './utils/PageLoading';
import { message, Modal, Form, Input, Button, Alert } from 'antd';
import { PiCrownSimpleDuotone } from "react-icons/pi";
import { getRandomAvatar } from '@/utils/randomavatar';
import { HiLogout } from "react-icons/hi";

const adminEmails = ['suryavmds7000@gmail.com', 'sivakcr8@gmail.com', 'sendiee.web@gmail.com', 'knithishseo@gmail.com'];

const OrgSelector = () => {
    const context = useContext(AppContext)
    const { update } = useSession();
    const router = useRouter();
    const [messageApi, contextHolder] = message.useMessage();
    const [newOrgModal, setNewOrgModal] = useState(false);
    const [loading, setLoading] = useState(false);
    const [formData, setFormData] = useState({
        name: '',
        description: ''
    });
    const [form] = Form.useForm();

    // Function to refresh user access in session
    const refreshUserAccess = async () => {
        await update();
    }

    // Function to store active organization in localStorage
    const storeActiveOrgInLS = (org) => {
        if (org && org._id) {
            localStorage.setItem('activeOrg', org._id);
        } else {
            localStorage.removeItem('activeOrg');
        }
    };

    // Function to handle organization selection
    const handleOrgSelect = (org) => {
        console.log('Selected Organization:', org);
        if (org && org._id) {
            context.setActiveOrg(org); // Set the selected organization in context
            storeActiveOrgInLS(org); // Store the selected organization in localStorage
            refreshUserAccess(); // Refresh user access to ensure the session is updated
            if(router.pathname === '/dashboard/selectorg') {
                router.push('/dashboard');
            }
        }
    };

    const checkAdminAccess = () => {
        if (context.userInfo && context.userInfo.email) {
            return adminEmails.includes(context.userInfo.email);
        }
        return false;
    };
  return (
    <div>
    {contextHolder}
    <PageLoading />
    <Modal
        title="Create New Organization"
        open={newOrgModal}
        onCancel={() => setNewOrgModal(false)}
        footer={null}
        width={600}
        centered
    >
        <Form
            form={form}
            style={{
                marginTop: '20px',
            }}
            layout="vertical"
            onFinish={async (values) => {
                setLoading(true);
                try {
                    const response = await fetch('/api/organization/create', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify(values)
                    });
                    const result = await response.json();
                    if (result.status === 'success') {
                        messageApi.success('Organization created successfully!');
                        context.setOrgList([...context.orgList, result.data]);
                        handleOrgSelect(result.data); // Select the newly created organization
                        storeActiveOrgInLS(result.data); // Store the new organization as active
                        refreshUserAccess(); // Refresh user access to ensure the session is updated
                        form.resetFields(); // Reset the form fields
                        setNewOrgModal(false);
                    } else {
                        messageApi.error(result.message || 'Failed to create organization.');
                    }
                } catch (error) {
                    console.error('Error creating organization:', error);
                    messageApi.error('An error occurred while creating the organization.');
                } finally {
                    setLoading(false);
                }
            }}
            initialValues={formData}
        >
            <Form.Item
                name="name"
                label=""
                rules={[{ required: true, message: 'Please enter organization name' }]}
            >
                <Input
                    placeholder="Enter organization name"
                    size='large'
                    autoFocus
                    value={formData.name}
                    minLength={5}
                    maxLength={100}
                    showCount
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                />
            </Form.Item>
            {!checkAdminAccess() && <Alert type='warning' message="Creating a new organization is currently restricted. Please contact +91 89460 42821 to create a new organization." style={{ marginBottom: '20px' }} />}
            {/* <Form.Item
                name="description"
                label="Description"
                rules={[{ required: true, message: 'Please enter organization description' }]}
            >
                <Input.TextArea
                    placeholder="Enter organization description"
                    rows={4}
                    onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                />
            </Form.Item> */}
            <Form.Item>
                <Button
                    block
                    type="primary"
                    htmlType="submit"
                    loading={loading}
                    size='large'
                    style={{ width: '100%' }}
                    disabled={!checkAdminAccess()}
                >
                    Create Organization
                </Button>
            </Form.Item>
        </Form>
    </Modal>
    <section className='org_selector_container'>
      <div className='org_selector'>
        <div className='org_selector_header'>
          <h3>Select Organization</h3>
        </div>
        <div className='org_selector_list'>
            {
                context.orgList && context.orgList.length > 0 ? 
                    context.orgList.map((org, index) => (
                    <div className='org_selector_item' key={index} onClick={() => handleOrgSelect(org)}>
                        <div className='org_selector_item_icon'>
                            <Image width={80} height={80} src={getRandomAvatar(org?.uniqueId)} alt='Avatar' />
                        </div>
                        <div className='org_selector_item_name'>{org.name} {org.isOwner && <span><PiCrownSimpleDuotone /></span>}</div>
                        <div className='org_selector_item_id'>
                            #{org.uniqueId}
                        </div>
                    </div>
                    )) : 
                    <div className='org_selector_empty'></div>
            }
        </div>
        <button className='org_selector_add_btn' onClick={() => setNewOrgModal(true)}>
            <span>Create Organization</span>
            <IoMdAdd />
        </button>
        <div style={{ textAlign: 'center', marginTop: '15px' }}>
            <Button type='link' danger onClick={() => {signOut()}} icon={<HiLogout />} iconPosition='end'>
                Logout
            </Button>
        </div>
      </div>

    </section>
    </div>
  )
}

export default OrgSelector