import { formatDate } from '@/utils/formatdate';
import { formatMilliCurrency } from '@/utils/formatmillicurrency';
import { Button, Card, Col, DatePicker, Form, InputNumber, message, Modal, Row, Select, Table, Tag, Typography } from 'antd';
import { useRouter } from 'next/router';
import React, { useEffect, useState } from 'react'

const { Title } = Typography;

const WalletComponent = () => {
    const router = useRouter();
    const { orgId } = router.query;
    const [loading, setLoading] = useState(false);
    const [messageApi, contextHolder] = message.useMessage();
    const [logs, setLogs] = useState([]);
    const [balance, setBalance] = useState(0);
    const [isModelOpen, setIsModelOpen] = useState(false);
    const [filterType, setFilterType] = useState('all');
    const [dateRange, setDateRange] = useState([null, null]);
    const [triggerEffects, setTriggerEffects] = useState(false);

    useEffect(() => {
            // Fetch wallet balance and logs when the component mounts
            const fetchWalletData = async () => {
                setLoading(true);
                try {
                    const response = await fetch('/api/admin/wallet/balance', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            orgId: orgId,
                        }),
                    });
                    const result = await response.json();
                    if (result.status === 'success') {
                        setBalance(result.balance);
                    } else {
                        messageApi.open({
                            type: 'error',
                            content: result.message || 'Failed to fetch wallet balance',
                        });
                    }
                } catch (error) {
                    messageApi.open({
                        type: 'error',
                        content: error.message || 'Failed to fetch wallet balance',
                    });
                } finally {
                    setLoading(false);
                }
            }
    
            fetchWalletData();
        }, [triggerEffects]);
    
        // Fetch wallet logs based on the selected filter type and date range
        const fetchWalletLogs = async () => {
            setLoading(true);
            //check if both dateRange[0] and dateRange[1] are selected
            if ( dateRange?.length && dateRange[0] && dateRange[1] && dateRange[0] > dateRange[1]) {
                return;
            }
            setLogs([]);
            try {
                const response = await fetch('/api/admin/wallet/logs', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        orgId: orgId,
                        filterType: filterType,
                        dateRange: dateRange,
                    }),
                });
                const result = await response.json();
                if (result.status === 'success') {
                    setLogs(result.logs);
                } else {
                    messageApi.open({
                        type: 'error',
                        content: result.message || 'Failed to fetch wallet logs',
                    });
                }
            } catch (error) {
                messageApi.open({
                    type: 'error',
                    content: error.message || 'Failed to fetch wallet logs',
                });
            } finally {
                setLoading(false);
            }
        }
    
        useEffect(() => {
            fetchWalletLogs();
        }, [filterType, dateRange, triggerEffects]);
  return (
    <div>
    {contextHolder}
        <Modal
            title="Add Funds"
            open={isModelOpen}
            onCancel={() => setIsModelOpen(false)}
            footer={null}
        >
            <Form
                layout="vertical"
                onFinish={async (values) => {
                    setLoading(true);
                    try {
                        const response = await fetch('/api/admin/wallet/add-funds', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                orgId: orgId,
                                amount: values.amount,
                            }),
                        });
                        const result = await response.json();
                        if (result.status === 'success') {
                            messageApi.open({
                                type: 'success',
                                content: 'Funds added successfully!',
                            });
                            setTriggerEffects(!triggerEffects);
                            setIsModelOpen(false);
                        } else {
                            messageApi.open({
                                type: 'error',
                                content: result.message || 'Failed to add funds',
                            });
                        }
                    } catch (error) {
                        messageApi.open({
                            type: 'error',
                            content: error.message || 'Failed to add funds',
                        });
                    } finally {
                        setLoading(false);
                    }
                }}
            >
                <Form.Item
                    label="Enter amount to add"
                    name="amount"
                    rules={[{ required: true, message: 'Please enter the amount to add' }]}
                >
                    <InputNumber style={{ width: '100%' }} placeholder="Enter amount" prefix="$" min={5} max={1000} />
                </Form.Item>
                <Form.Item>
                    <Button type="primary" htmlType="submit" loading={loading}>
                        Add Funds
                    </Button>
                </Form.Item>
            </Form>
        </Modal>
        <Row gutter={[15, 15]}>
            <Col xs={24} lg={12}>
            {/* Card with wallet balance and add funds button */}
            <Card className='card'>
                <Row gutter={[15, 15]}>
                    <Col span={12}>
                        <Title style={{ marginBottom: '0' }} level={4}>Balance: {formatMilliCurrency(balance || 0)}</Title>
                    </Col>
                    <Col span={12} style={{textAlign: 'right'}}>
                        <Button type="" onClick={() => setIsModelOpen(true)}>Add Funds</Button>
                    </Col>
                </Row>
            </Card>
            </Col>
            <Col span={24}>
            {/* Card with wallet logs and filter options */}
            <Card className='card' title="Wallet Logs">
                <Row gutter={[15, 15]} style={{ marginBottom: '15px' }}>
                    <Col xs={24} sm={12} md={8}>
                        <Select
                            value={filterType}
                            onChange={(value) => setFilterType(value)}
                            style={{ width: '100%' }}
                            options={[
                                { label: 'All', value: 'all' },
                                { label: 'Credits', value: 'credit' },
                                { label: 'Debits', value: 'debit' },
                            ]}
                        />
                    </Col>
                    <Col xs={24} sm={12} md={8}>
                        <DatePicker.RangePicker
                            value={dateRange}
                            onChange={(dates) => setDateRange(dates)}
                            format="DD-MM-YYYY"
                            style={{ width: '100%' }}
                        />
                    </Col>
                    <Col span={24} style={{ marginTop: '15px' }}>
                        <div className='table_container'>
                            <Table
                                dataSource={logs}
                                loading={loading}
                                rowKey="_id"
                                pagination={{ pageSize: 10 }}
                                columns={
                                    [
                                        {
                                            title: 'Date',
                                            dataIndex: 'timestamp',
                                            key: 'timestamp',
                                            render: (text) => formatDate(text),
                                        },
                                        {
                                            title: 'Type',
                                            dataIndex: 'type',
                                            key: 'type',
                                            render: (text) => (
                                                <Tag color={text === 'CREDIT' ? 'green' : 'warning'}>
                                                    {text.charAt(0).toUpperCase() + text.slice(1)}
                                                </Tag>
                                            ),
                                        },
                                        {
                                            title: 'Amount',
                                            dataIndex: 'amountConsumed',
                                            key: 'amountConsumed',
                                            render: (text) => formatMilliCurrency(text || 0),
                                        },
                                        {
                                            title: 'Description',
                                            dataIndex: 'description',
                                            key: 'description',
                                            render: (_, record) => {
                                                if(record.category === 'AI') {
                                                    return <span>
                                                        Amount consumed for AI Usage
                                                    </span>;
                                                } else if(record.category === 'ADD_FUNDS') {
                                                    return <span>
                                                        Amount added to wallet
                                                    </span>;
                                                } else if(record.category === 'SUBSCRIPTION') {
                                                    return <span>
                                                        Amount consumed for Subscription
                                                    </span>;
                                                } else {
                                                    return <span>
                                                        {''}
                                                    </span>;
                                                }
                                            },
                                        },

                                    ]
                                }
                            >
                            </Table>
                        </div>
                    </Col>
                </Row>
            </Card>
            </Col>
        </Row>
    </div>
  )
}

export default WalletComponent