import { Avatar, Button, Col, Drawer, Flex, Form, Input, Popconfirm, Row, Select, Space, Switch, message } from 'antd';
import React, { useState, useEffect } from 'react';

const ManageUsersDrawer = ({ isOpen, showDrawer, triggerSuccess, userData, setUserData }) => {
    const [messageApi, contextHolder] = message.useMessage();
    const [formValues, setFormValues] = useState(userData || {
        isActive: true,
    });
    const [isSubmitLoading, setSubmitLoading] = useState(false);
    const [form] = Form.useForm();
    const [isDeleteLoading, setDeleteLoading] = useState(false)

    useEffect(() => {
        if (userData) {
            setFormValues({
                ...userData,
                password: ''
            })
            form.setFieldsValue({
                isActive: userData.isActive,
                name: userData.name,
                email: userData.email,
                password: '',
            });
        } else {
            form.resetFields();
        }
    }, [userData]);

    useEffect(() => {
        if(!isOpen){
            form.resetFields();
            setFormValues({
                isActive: true,
                name: '',
                email: '',
                password: '',
            })
            setUserData(null)
        }
    }, [isOpen])

    const onClose = () => {
        showDrawer(false);
    };

    const onSubmit = async () => {
        setSubmitLoading(true);
        try {
            const values = await form.validateFields();
            try {
                const response = await fetch(userData ? '/api/admin/users/update' : '/api/admin/users/create', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ ...formValues })
                });
                const data = await response.json();
                setSubmitLoading(false);
                if (data.status === 'success') {
                    messageApi.open({
                        type: 'success',
                        content: `User ${userData ? 'updated' : 'added'} successfully!`,
                    });
                    setFormValues({
                        isActive: true,
                        name: '',
                        email: '',
                        password: '',
                    });
                    form.resetFields();
                    form.setFieldsValue({
                        isActive: true,
                        name: '',
                        email: '',
                        password: '',
                    });
                    triggerSuccess(true);
                    showDrawer(false);
                } else {
                    messageApi.open({
                        type: 'error',
                        content: data.message,
                    });
                }
            } catch (err) {
                console.log(err);
                messageApi.open({
                    type: 'error',
                    content: `Failed to ${userData ? 'update' : 'create'} user`,
                });
                setSubmitLoading(false);
            }
        } catch (err) {
            console.log('Validation error', err);
            setSubmitLoading(false);
        }
    };

    const onInputChangeHandler = (event) => {
        const { name, value } = event.target;
        setFormValues({
            ...formValues,
            [name]: value
        });
    };

    const handleSelectChange = (value, name) => {
        setFormValues({
            ...formValues,
            [name]: value
        });
    };

    const deleteUser = async () => {
        try{
            setDeleteLoading(true)
            const response = await fetch('/api/admin/users/delete', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({...userData})
            });
            const data = await response.json();
            setDeleteLoading(false)
            if(data.status === 'success'){
                messageApi.open({
                    type: 'success',
                    content: 'User removed successfully!',
                });
                triggerSuccess(true)
                showDrawer(false)
            }else{
                messageApi.open({
                    type: 'error',
                    content: data.message,
                });
                setDeleteLoading(false)
            }

        }catch(err){
            console.log(err)
            messageApi.open({
                type: 'error',
                content: 'Failed to delete user',
            });
        }
    }

    return (
        <>
            {contextHolder}
            <Drawer
                title={`${userData ? 'Update' : 'Add'} user`}
                width={720}
                onClose={onClose}
                open={isOpen}
                style={{
                    paddingBottom: 80,
                }}
                extra={
                    <Space>
                        {/* <Button onClick={onClose}>Cancel</Button> */}
                        {
                            userData &&(
                                <Popconfirm
                                    title="Delete user?"
                                    description="This user will be deleted permanently"
                                    okText="Delete"
                                    okType="danger"
                                    cancelText="Cancel"
                                    onConfirm={() => deleteUser()}>
                                    <Button loading={isDeleteLoading} danger>Delete user</Button>
                                </Popconfirm>
                            )
                        }
                        <Button onClick={onSubmit} loading={isSubmitLoading} type="primary">
                            {userData ? 'Update' : 'Add'} User
                        </Button>
                    </Space>
                }
            >
                <Form layout="vertical" form={form}>
                    <Row gutter={16}>
                        <Col span={24}>
                            <Flex align='center' justify='center' gap={5} style={{ marginBottom: '15px', width: '100%' }}>
                                <Avatar size={150} shape='square' src={<img src={formValues.profilePicture || 'https://www.gravatar.com/avatar/00000000000000000000000000000000?d=mp&f=y'}></img>} />
                            </Flex>
                        </Col>
                        <Col span={24}>
                            <Form.Item
                                name="name"
                                label="Name"
                            >
                                <Input
                                    placeholder="Enter name"
                                    name="name"
                                    value={formValues.name}
                                    onChange={onInputChangeHandler}
                                />
                            </Form.Item>
                        </Col>
                        <Col span={24}>
                            <Form.Item
                                name="email"
                                label="Email"
                                rules={[{ required: true, type: 'email', message: 'Please enter a valid email' }]}
                            >
                                <Input
                                    placeholder="Enter email"
                                    name="email"
                                    value={formValues.email}
                                    allowClear
                                    type="email"
                                    onChange={onInputChangeHandler}
                                />
                            </Form.Item>
                        </Col>
                        <Col span={24}>
                            <Form.Item
                                name="password"
                                label="Password"
                            >
                                <Input.Password
                                    placeholder="Enter password"
                                    name="password"
                                    value={formValues.password}
                                    onChange={onInputChangeHandler}
                                />
                            </Form.Item>
                        </Col>
                        <Col span={12}>
                            <Form.Item
                                name="isActive"
                                label="Is Active?"
                            >
                                <Switch
                                    checked={formValues.isActive}
                                    onChange={(checked) => handleSelectChange(checked, 'isActive')}
                                />
                            </Form.Item>
                        </Col>
                    </Row>
                </Form>
            </Drawer>
        </>
    );
};

export default ManageUsersDrawer;
