import { ALL_MODELS } from '@/constant/variables';
import { Button, Col, Drawer, Form, Input, InputNumber, Popconfirm, Row, Select, Space, message } from 'antd';
import React, { useState, useEffect } from 'react';

const PricingManagementDrawer = ({ isOpen, showDrawer, triggerSuccess, pricingData, setPricingData }) => {
    const [messageApi, contextHolder] = message.useMessage();
    const [formValues, setFormValues] = useState(pricingData || {});
    const [isSubmitLoading, setSubmitLoading] = useState(false);
    const [form] = Form.useForm();
    const [isDeleteLoading, setDeleteLoading] = useState(false)

    useEffect(() => {
        if (pricingData) {
            setFormValues({
                ...pricingData,
                input: pricingData?.pricing?.input || '',
                output: pricingData?.pricing?.output || '',
                cache: pricingData?.pricing?.cache || ''
            })
            form.setFieldsValue({
                model: pricingData.model,
                input: pricingData?.pricing?.input || '',
                output: pricingData?.pricing?.output || '',
                cache: pricingData?.pricing?.cache || '',
            });
        } else {
            form.resetFields();
        }
    }, [pricingData]);

    useEffect(() => {
        if(!isOpen){
            form.resetFields();
            setFormValues({})
            setPricingData(null)
        }
    }, [isOpen])

    const onClose = () => {
        showDrawer(false);
    };

    const onSubmit = async () => {
        setSubmitLoading(true);
        try {
            const values = await form.validateFields();
            try {
                const response = await fetch(pricingData ? '/api/admin/pricing/update' : '/api/admin/pricing/create', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ ...formValues })
                });
                const data = await response.json();
                setSubmitLoading(false);
                if (data.status === 'success') {
                    messageApi.open({
                        type: 'success',
                        content: `Cost ${pricingData ? 'updated' : 'added'} successfully!`,
                    });
                    setFormValues({});
                    form.resetFields();
                    form.setFieldsValue({
                        model: '',
                        input: '',
                        output: '',
                        cache: ''
                    });
                    triggerSuccess(true);
                    showDrawer(false);
                } else {
                    messageApi.open({
                        type: 'error',
                        content: data.message,
                    });
                }
            } catch (err) {
                console.log(err);
                messageApi.open({
                    type: 'error',
                    content: `Failed to ${pricingData ? 'update' : 'create'} cost`,
                });
                setSubmitLoading(false);
            }
        } catch (err) {
            console.log('Validation error', err);
            setSubmitLoading(false);
        }
    };

    const onInputChangeHandler = (event) => {
        const { name, value } = event.target;
        setFormValues({
            ...formValues,
            [name]: value
        });
    };

    const handleSelectChange = (value, name) => {
        setFormValues({
            ...formValues,
            [name]: value
        });
    };

    const deleteServer = async () => {
        try{
            setDeleteLoading(true)
            const response = await fetch('/api/admin/pricing/delete', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({...pricingData})
            });
            const data = await response.json();
            setDeleteLoading(false)
            if(data.status === 'success'){
                messageApi.open({
                    type: 'success',
                    content: 'Cost data deleted successfully!',
                });
                triggerSuccess(true)
                showDrawer(false)
            }else{
                messageApi.open({
                    type: 'error',
                    content: data.message,
                });
                setDeleteLoading(false)
            }

        }catch(err){
            console.log(err)
            messageApi.open({
                type: 'error',
                content: 'Failed to delete cost data',
            });
        }
    }

    return (
        <>
            {contextHolder}
            <Drawer
                title={`${pricingData ? 'Update' : 'Add'} cost`}
                width={720}
                onClose={onClose}
                open={isOpen}
                style={{
                    paddingBottom: 80,
                }}
                extra={
                    <Space>
                        {/* <Button onClick={onClose}>Cancel</Button> */}
                        {
                            pricingData &&(
                                <Popconfirm
                                    title="Delete cost?"
                                    description="This will be deleted permanently"
                                    okText="Delete"
                                    okType="danger"
                                    cancelText="Cancel"
                                    onConfirm={() => deleteServer()}>
                                    <Button loading={isDeleteLoading} danger>Delete cost</Button>
                                </Popconfirm>
                            )
                        }
                        <Button onClick={onSubmit} loading={isSubmitLoading} type="primary">
                            {pricingData ? 'Update' : 'Add'} Cost
                        </Button>
                    </Space>
                }
            >
                <Form layout="vertical" form={form}>
                    <Row gutter={[16, 16]}>
                        <Col span={24}>
                            <Form.Item
                                name="model"
                                label="Model"
                                rules={[{ required: true, message: 'Please select a model' }]}
                            >
                                <Select
                                    placeholder="Select a model"
                                    value={formValues.model}
                                    onChange={(value) => handleSelectChange(value, 'model')}
                                    options={ALL_MODELS.map(model => ({ label: model?.value, value: model?.value }))}
                                />
                            </Form.Item>
                        </Col>
                        <Col lg={24} xs={24}>
                            <Form.Item
                                name="input"
                                label="Input Cost"
                                rules={[{ required: true, message: 'Please enter input cost' }]}
                            >
                                <InputNumber
                                    prefix="$"
                                    suffix="/1M Tokens"
                                    min={0}
                                    max={1000}
                                    step={0.01}
                                    placeholder="Enter input cost"
                                    style={{ width: '100%' }}
                                    value={formValues.input}
                                    onChange={(value) => handleSelectChange(value, 'input')}
                                />
                            </Form.Item>
                        </Col>
                        <Col lg={24} xs={24}>
                            <Form.Item
                                name="output"
                                label="Output Cost"
                                rules={[{ required: true, message: 'Please enter output cost' }]}
                            >
                                <InputNumber
                                    prefix="$"
                                    suffix="/1M Tokens"
                                    min={0}
                                    max={1000}
                                    step={0.01}
                                    placeholder="Enter output cost"
                                    style={{ width: '100%' }}
                                    value={formValues.output}
                                    onChange={(value) => handleSelectChange(value, 'output')}
                                />
                            </Form.Item>
                        </Col>
                        <Col lg={24} xs={24}>
                            <Form.Item
                                name="cache"
                                label="Cache Cost"
                                rules={[{ required: false }]}
                            >
                                <InputNumber
                                    prefix="$"
                                    suffix="/1M Tokens"
                                    min={0}
                                    max={1000}
                                    step={0.01}
                                    placeholder="Enter cache cost (optional)"
                                    style={{ width: '100%' }}
                                    value={formValues.cache}
                                    onChange={(value) => handleSelectChange(value, 'cache')}
                                />
                            </Form.Item>
                        </Col>
                    </Row>
                </Form>
            </Drawer>
        </>
    );
};

export default PricingManagementDrawer;
