import { Button, Col, Drawer, Form, Input, Popconfirm, Row, Select, Space, Switch, message } from 'antd';
import React, { useState, useEffect } from 'react';

const ServerManagementDrawer = ({ isOpen, showDrawer, triggerSuccess, serverData, setServerData }) => {
    const [messageApi, contextHolder] = message.useMessage();
    const [formValues, setFormValues] = useState(serverData || {});
    const [isSubmitLoading, setSubmitLoading] = useState(false);
    const [form] = Form.useForm();
    const [orgsOptions, setOrgsOptions] = useState([]);
    const [isDeleteLoading, setDeleteLoading] = useState(false)

    useEffect(() => {
        const fetchOrgs = async () => {
            try {
                const response = await fetch('/api/admin/orgs/list?type=active');
                const data = await response.json();
                if (data.status === 'success') {
                    setOrgsOptions(data?.data?.map(org => ({
                        label: org.name,
                        value: org._id
                    })));
                }
            } catch (error) {
                console.error('Error fetching orgs:', error);
            }
        };

        fetchOrgs();

        if (serverData) {
            setFormValues({
                ...serverData,
                orgId: serverData?.orgId?._id || null,
            })
            form.setFieldsValue({
                serverUrl: serverData.serverUrl,
                notes: serverData.notes,
                orgId: serverData?.orgId?._id || null,
                maxLimits: serverData.maxLimits,
                isActive: serverData.isActive,
            });
        } else {
            form.resetFields();
        }
    }, [serverData]);

    useEffect(() => {
        if(!isOpen){
            form.resetFields();
            setFormValues({})
            setServerData(null)
        }
    }, [isOpen])

    const onClose = () => {
        showDrawer(false);
    };

    const onSubmit = async () => {
        setSubmitLoading(true);
        try {
            const values = await form.validateFields();
            try {
                const response = await fetch(serverData ? '/api/admin/servers/update' : '/api/admin/servers/create', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ ...formValues })
                });
                const data = await response.json();
                setSubmitLoading(false);
                if (data.status === 'success') {
                    messageApi.open({
                        type: 'success',
                        content: `Server ${serverData ? 'updated' : 'added'} successfully!`,
                    });
                    setFormValues({});
                    form.resetFields();
                    form.setFieldsValue({
                        serverUrl: '',
                        notes: '',
                        orgId: '',
                        maxLimits: '',
                        isActive: true,
                    });
                    triggerSuccess(true);
                    showDrawer(false);
                } else {
                    messageApi.open({
                        type: 'error',
                        content: data.message,
                    });
                }
            } catch (err) {
                console.log(err);
                messageApi.open({
                    type: 'error',
                    content: `Failed to ${serverData ? 'update' : 'create'} server`,
                });
                setSubmitLoading(false);
            }
        } catch (err) {
            console.log('Validation error', err);
            setSubmitLoading(false);
        }
    };

    const onInputChangeHandler = (event) => {
        const { name, value } = event.target;
        setFormValues({
            ...formValues,
            [name]: value
        });
    };

    const handleSelectChange = (value, name) => {
        setFormValues({
            ...formValues,
            [name]: value
        });
    };

    const deleteServer = async () => {
        try{
            setDeleteLoading(true)
            const response = await fetch('/api/admin/servers/delete', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({...serverData})
            });
            const data = await response.json();
            setDeleteLoading(false)
            if(data.status === 'success'){
                messageApi.open({
                    type: 'success',
                    content: 'Server removed successfully!',
                });
                triggerSuccess(true)
                showDrawer(false)
            }else{
                messageApi.open({
                    type: 'error',
                    content: data.message,
                });
                setDeleteLoading(false)
            }

        }catch(err){
            console.log(err)
            messageApi.open({
                type: 'error',
                content: 'Failed to delete server',
            });
        }
    }

    return (
        <>
            {contextHolder}
            <Drawer
                title={`${serverData ? 'Update' : 'Add'} server`}
                width={720}
                onClose={onClose}
                open={isOpen}
                style={{
                    paddingBottom: 80,
                }}
                extra={
                    <Space>
                        {/* <Button onClick={onClose}>Cancel</Button> */}
                        {
                            serverData &&(
                                <Popconfirm
                                    title="Delete server?"
                                    description="This server will be deleted permanently"
                                    okText="Delete"
                                    okType="danger"
                                    cancelText="Cancel"
                                    onConfirm={() => deleteServer()}>
                                    <Button loading={isDeleteLoading} danger>Delete server</Button>
                                </Popconfirm>
                            )
                        }
                        <Button onClick={onSubmit} loading={isSubmitLoading} type="primary">
                            {serverData ? 'Update' : 'Add'} Server
                        </Button>
                    </Space>
                }
            >
                <Form layout="vertical" form={form}>
                    <Row gutter={16}>
                        <Col span={24}>
                            <Form.Item
                                name="serverUrl"
                                label="Server URL"
                                rules={[{ required: true, message: 'Please enter server URL' },
                                    { type: 'url', message: 'Please enter a valid URL' },
                                    { pattern: /^(?!.*\/$).+/, message: 'URL should not end with a slash' }
                                ]}
                            >
                                <Input
                                    placeholder="Enter server URL"
                                    name="serverUrl"
                                    value={formValues.serverUrl}
                                    onChange={onInputChangeHandler}
                                />
                            </Form.Item>
                        </Col>
                        <Col span={24}>
                            <Form.Item
                                name="notes"
                                label="Notes"
                            >
                                <Input.TextArea
                                    placeholder="Enter notes (recommended)"
                                    name="notes"
                                    value={formValues.notes}
                                    onChange={onInputChangeHandler}
                                />
                            </Form.Item>
                        </Col>
                        <Col span={24}>
                            <Form.Item
                                name="orgId"
                                label="Organization"
                                tooltip="Select organization if this server is specific to an organization"
                            >
                                <Select
                                    placeholder="Select organization"
                                    options={orgsOptions}
                                    allowClear
                                    value={formValues.orgId}
                                    onChange={(value) => handleSelectChange(value, 'orgId')}
                                />
                            </Form.Item>
                        </Col>
                        <Col span={12}>
                            <Form.Item
                                name="maxLimits"
                                label="Max Usage Limit"
                                tooltip="Maximum number of whatsapp instances allowed"
                                rules={[{ required: true, message: 'Please enter max limit' }]}
                            >
                                <Input
                                    type="number"
                                    placeholder="Enter max limit"
                                    name="maxLimits"
                                    minLength={0}
                                    maxLength={50}
                                    value={formValues.maxLimits}
                                    onChange={onInputChangeHandler}
                                />
                            </Form.Item>
                        </Col>
                        <Col span={12}>
                            <Form.Item
                                name="isActive"
                                label="Is Active?"
                            >
                                <Switch
                                    checked={formValues.isActive}
                                    onChange={(checked) => handleSelectChange(checked, 'isActive')}
                                />
                            </Form.Item>
                        </Col>
                    </Row>
                </Form>
            </Drawer>
        </>
    );
};

export default ServerManagementDrawer;
