import mongoose from 'mongoose';

const organizationSchema = new mongoose.Schema({
    name: {
        type: String,
        required: [true, 'Organization name is required'],
        maxlength: [100, 'Name cannot exceed 100 characters'],
        trim: true,
    },
    uniqueId: {
        type: String,
        required: [true, 'Unique ID is required'],
        unique: true,
        maxlength: [50, 'Unique ID cannot exceed 50 characters'],
        trim: true,
    },
    description: {
        type: String,
        required: false,
        trim: true,
        maxlength: [500, 'Description cannot exceed 500 characters'],
    },
    logoUrl: {
        type: String,
        required: false,
        trim: true,
    },
    ownerId: {
        type: mongoose.Schema.Types.ObjectId,
        ref: 'User',
        required: [true, 'Owner ID is required'],
    },
    members: [{
        userId: { type: mongoose.Schema.Types.ObjectId, ref: 'User' },
        createdAt: {
            type: Date,
            default: Date.now,
        },
        updatedAt: {
            type: Date,
            default: Date.now,
        },
        access: [{
            type: String,
            enum: ['read', 'write', 'admin', 'limited'],
            default: 'read',
        }],
        isActive: {
            type: Boolean,
            default: true,
        },
        _id: false,
    }],
    isActive: {
        type: Boolean,
        default: true,
    },
    createdAt: {
        type: Date,
        default: Date.now,
    },
    updatedAt: {
        type: Date,
        default: Date.now,
    },
}, {
    timestamps: true,
    versionKey: false,
});

// Compound index: ensures unique name per ownerId
organizationSchema.index({ ownerId: 1, name: 1 }, { unique: true });

// Middleware to update timestamps
organizationSchema.pre('save', function (next) {
    if (this.isModified()) {
        this.updatedAt = Date.now();
    }
    next();
});

// Middleware to handle member updates
organizationSchema.pre('findOneAndUpdate', function (next) {
    const update = this.getUpdate();
    if (update && update.$set) {
        if (update.$set.members) {
            update.$set.members.forEach(member => {
                member.updatedAt = Date.now();
            });
        }
        update.$set.updatedAt = Date.now();
    }
    next();
});

// Middleware to handle member removal
organizationSchema.pre('findOneAndRemove', function (next) {
    const update = this.getUpdate();
    if (update && update.$set) {
        if (update.$set.members) {
            update.$set.members.forEach(member => {
                member.updatedAt = Date.now();
            });
        }
        update.$set.updatedAt = Date.now();
    }
    next();
});

const Organization = mongoose.models.Organization || mongoose.model('Organization', organizationSchema);
export default Organization;