import AppContext from '@/context/AppContext';
import { Button, message, Modal, QRCode } from 'antd';
import React, { useContext, useEffect, useState } from 'react'

const LinkWhatsAppModal = ({
    isOpen,
    onClose,
    onLinkWhatsApp,
    selectedInstance,
    setSelectedInstance
}) => {
    const [loading, setLoading] = useState(false);
    const [qrLoading, setQrLoading] = useState(true);
    const [deleteLoading, setDeleteLoading] = useState(false);
    const [qrStatus, setQrStatus] = useState('loading'); // loading, active, scanned, expired
    const [qrData, setQrData] = useState(null);
    const [instanceId, setInstanceId] = useState(null);
    const [serverUrl, setServerUrl] = useState(null);
    const context = useContext(AppContext);
    const [messageApi, contextHolder] = message.useMessage();
    const [triggerRefresh, setTriggerRefresh] = useState(false);
    const [fetchCount, setFetchCount] = useState(0);

    // Handle modal close
    const handleClose = async () => {
         try {
            setDeleteLoading(true);
            if(!selectedInstance?.instance_id && instanceId) {
                const response = await fetch('/api/instance/delete', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        instance_id: instanceId,
                        serverUrl: serverUrl,
                        orgId: context?.activeOrg?._id,
                    }),
                });
            }
        } catch (error) {
            console.log('Error deleting WhatsApp instance:', error);
        } finally {
            setDeleteLoading(false);
            setLoading(false);
            onClose(false);
            setQrStatus('loading');
            setQrData(null);
            setInstanceId(null);
            setQrLoading(true);
            onLinkWhatsApp(false);
            setSelectedInstance(null);
        }
    }

    const handleDone = () => {
        setLoading(false);
        onClose(false);
        setQrStatus('loading');
        setQrData(null);
        setInstanceId(null);
        setQrLoading(true);
        onLinkWhatsApp(false);
        setSelectedInstance(null);
    }

    // Reset fetchCount when QR is refreshed
    useEffect(() => {
        setFetchCount(0);
    }, [triggerRefresh, instanceId]);

    // Handle Modal onOpen
    useEffect(() => {
        if (isOpen && selectedInstance?.instance_id) {
            setInstanceId(selectedInstance.instance_id);
            setServerUrl(selectedInstance.serverUrl);
            setQrData(selectedInstance.QRCode);
            setQrStatus(selectedInstance.status);
        }
    }, [isOpen]);

    // Create QR Code instance when the modal is opened
    useEffect(() => {
        const fetchQRCode = async () => {
            setQrLoading(true);
            setQrStatus('loading');
            setInstanceId(null);
            try {
                // Simulate fetching QR code data
                const response = await fetch('/api/instance/create', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        orgId: context?.activeOrg?._id,
                    }),
                });
                const data = await response.json();
                if (data.status === 'success') {
                    setInstanceId(data.data.instance_id);
                    setServerUrl(data.data.serverUrl);
                } else {
                    messageApi.open({
                        type: 'error',
                        content: data.message || 'Failed to fetch QR code',
                    });
                    setQrStatus('expired');
                }
            } catch (error) {
                console.error('Error fetching QR code:', error);
                messageApi.open({
                    type: 'error',
                    content: 'Failed to fetch QR code',
                });
                setQrStatus('expired');
            } finally {
                setQrLoading(false);
            }
        }

        if (isOpen && !selectedInstance?.instance_id && !instanceId) {
            fetchQRCode();
        }
    }, [isOpen, triggerRefresh])

    // Handle QR code status refresh for every 2 seconds when the instanceId is available
    useEffect(() => {
        let interval;
        if (instanceId && qrStatus !== 'expired' && qrStatus !== 'scanned' && deleteLoading === false) {
            interval = setInterval(async () => {
                setFetchCount(prev => {
                    if (prev >= 5) { // 6th fetch (0-based)
                        setQrStatus('expired');
                        clearInterval(interval);
                        return prev;
                    }
                    return prev + 1;
                });
            if (fetchCount >= 5) return; // Prevent extra fetch after expired
                try {
                    const response = await fetch(`/api/instance/status`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            instanceId: instanceId,
                            orgId: context?.activeOrg?._id,
                        }),
                    });
                    const data = await response.json();
                    if (data.status === 'success') {
                        setQrStatus(data.data.status);
                        if(data.data.status === 'active') {
                            setQrData(data.data.QRCode);
                        }
                        if (data.data.status === 'scanned') {
                            messageApi.open({
                                type: 'success',
                                content: 'WhatsApp account linked successfully!',
                            });
                            onLinkWhatsApp(true);
                            handleDone();
                        }
                    } else {
                        messageApi.open({
                            type: 'error',
                            content: data.message || 'Failed to fetch QR code status',
                        });
                        setQrStatus('expired');
                    }
                } catch (error) {
                    console.error('Error fetching QR code status:', error);
                    messageApi.open({
                        type: 'error',
                        content: 'Failed to fetch QR code status',
                    });
                    setQrStatus('expired');
                }
            }, 5000);
        }
        return () => {
            if (interval) {
                clearInterval(interval);
            }
        }
    }, [instanceId, qrStatus, fetchCount, context?.activeOrg?._id]);

    return (
        <Modal
            title="Link WhatsApp Account"
            open={isOpen}
            onOk={() => handleDone()}
            onCancel={() => handleClose()}
            maskClosable={false}
            closeIcon={null}
            footer={[
                <Button key="back" onClick={() => handleClose()} danger loading={deleteLoading}>
                    {!selectedInstance?.instance_id ? 'Discard' : 'Close'}
                </Button>,
                <Button key="submit" type="primary" loading={loading} onClick={() => handleDone()} disabled={deleteLoading}>
                    Done
                </Button>,
            ]}
        >
            {contextHolder}
            <div className='qr_container'>
                <QRCode size={250} status={qrStatus} value={
                    qrData ? qrData : '#SENDIEE-QR-CODE-NOT-AVAILABLE#'
                }
                onRefresh={() => {setQrStatus('loading'); setTriggerRefresh(!triggerRefresh)}}
                />
                <p>Open WhatsApp on your phone, Tap Menu or Settings and select Linked Devices. Scan this QR to connect</p>
            </div>
        </Modal>
    )
}

export default LinkWhatsAppModal