import withAdminAuth from '@/components/auth/withAdminAuth';
import DashboardLayout from '@/components/layout/DashboardLayout';
import PageLoading from '@/components/utils/PageLoading';
import { Breadcrumb, Button, Card, Col, Flex, message, Row, Space, Table, Tag } from 'antd';
import Head from 'next/head';
import Link from 'next/link';
import React, { useState, useEffect } from 'react'
import { RiPencilFill } from "react-icons/ri";
import { useRouter } from 'next/router';
import { FaArrowRightLong } from "react-icons/fa6";
import Avatar from 'antd/es/avatar/Avatar';
import { getRandomAvatar } from '@/utils/randomavatar';
import { formatMilliCurrency } from '@/utils/formatmillicurrency';

const OrgsAdminPage = () => {
    const [loading, setLoading] = useState(false);
    const [data, setData] = useState([]);
    const [wallets, setWallets] = useState([]);
    const [triggerEffects, setTriggerEffects] = useState(false);
    const [messageApi, contextHolder] = message.useMessage();
    const router = useRouter();

    useEffect(() => {
        const fetchOrgs = async () => {
            setLoading(true);
            try {
                const response = await fetch('/api/admin/orgs/list');
                const result = await response.json();
                if (result.status === 'success') {
                    setData(result.data);
                    setWallets(result.wallets || []);
                } else {
                    messageApi.error(result.message || 'Failed to fetch orgs');
                }
            } catch (error) {
                console.error('Error fetching orgs:', error);
                messageApi.error('Failed to fetch orgs');
            } finally {
                setLoading(false);
            }
        };

        fetchOrgs();
    }, [triggerEffects]);

  return (
    <>
      <Head>
        <title>Orgs | Sendiee Admin</title>
        <meta name="viewport" content="width=device-width, initial-scale=1" />
      </Head>
      <PageLoading />
      {contextHolder}
      <main>
        <DashboardLayout title="Organizations" loading={false} activemenu="orgs">
            <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
                <Breadcrumb
                    items={[
                    {
                        title:<Link href={"/dashboard"}>Dashboard</Link>,
                    },
                    {
                        title: 'Organizations',
                    },
                    ]}
                />
            </Space>
            <Row gutter={[15, 15]}>
                <Col span={24}>
                <Card title="Org Management"
                >
                    <div className='table_container'>
                    <Table
                        dataSource={data}
                        rowKey="_id"
                        pagination={{ pageSize: 10 }}
                        columns={[
                            {
                                title: 'Name',
                                dataIndex: 'name',
                                key: 'name',
                                render: (_, record) => (
                                    <Flex align='center' gap={5}>
                                        <Avatar shape='square' src={<img src={getRandomAvatar(record.uniqueId)}></img>} />
                                        {/* <Button type="link" onClick={() => router.push(`/admin/orgs/${record._id}`)}> */}
                                            {record.name}
                                        {/* </Button> */}
                                    </Flex>
                                ),
                            },
                            {
                                title: 'UID',
                                dataIndex: 'uniqueId',
                                key: 'uniqueId',
                            },
                            {
                                title: 'Status',
                                dataIndex: 'isActive',
                                key: 'isActive',
                                render: (text) => (
                                    <Tag color={text ? 'green' : 'yellow'}>
                                        {text ? 'Active' : 'Inactive'}
                                    </Tag>
                                ),
                            },
                            {
                                title: 'Owner',
                                dataIndex: 'ownerId',
                                key: 'ownerId',
                                render: (text) => text ? <Tag>{text.email}</Tag> : '-',
                            },
                            {
                                title: 'Balance',
                                key: 'balance',
                                render: (_, record) => {
                                    const wallet = wallets.find(wallet => wallet.orgId === record._id);
                                    return <span>{formatMilliCurrency(wallet ? wallet.balance : 0)}</span>;
                                }
                            },
                            {
                                title: 'Created At',
                                dataIndex: 'createdAt',
                                key: 'createdAt',
                                render: (text) => new Date(text).toLocaleDateString(),
                            },
                            {
                                title: 'Actions',
                                key: 'actions',
                                render: (_, record) => (
                                    <Space size="middle">
                                        <Button size='small' onClick={() => {
                                            router.push(`/admin/orgs/${record._id}`);
                                        }}><FaArrowRightLong /></Button>
                                    </Space>
                                ),
                            },
                        ]}
                        loading={loading}
                        bordered
                        size="small"
                    />
                    </div>
                </Card>
                </Col>
            </Row>
        </DashboardLayout>
      </main>
    </>
  )
}

export default withAdminAuth(OrgsAdminPage);