import withAdminAuth from '@/components/auth/withAdminAuth';
import DashboardLayout from '@/components/layout/DashboardLayout';
import PageLoading from '@/components/utils/PageLoading';
import { Breadcrumb, Button, Card, Col, message, Row, Space, Table, Tag } from 'antd';
import Head from 'next/head';
import Link from 'next/link';
import React, { useState, useEffect } from 'react'
import { RiPencilFill } from "react-icons/ri";
import { LuTrash2 } from "react-icons/lu";
import ServerManagementDrawer from '@/drawers/ServersManagementDrwer';
import PricingManagementDrawer from '@/drawers/PricingMagementDrawer';

const PricingAdminPage = () => {
    const [loading, setLoading] = useState(false);
    const [data, setData] = useState([]);
    const [triggerEffects, setTriggerEffects] = useState(false);
    const [messageApi, contextHolder] = message.useMessage();
    const [isManageDrawerOpen, setIsManageDrawerOpen] = useState(false);
    const [selectedServer, setSelectedServer] = useState(null);

    useEffect(() => {
        const fetchServers = async () => {
            setLoading(true);
            try {
                const response = await fetch('/api/admin/pricing/list', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                });
                const result = await response.json();
                if (result.status === 'success') {
                    setData(result.data);
                } else {
                    messageApi.error(result.message || 'Failed to fetch costs');
                }
            } catch (error) {
                console.error('Error fetching costs:', error);
                messageApi.error('Failed to fetch costs');
            } finally {
                setLoading(false);
            }
        };

        fetchServers();
    }, [triggerEffects]);

  return (
    <>
      <Head>
        <title>Model Cost | Sendiee Admin</title>
        <meta name="viewport" content="width=device-width, initial-scale=1" />
      </Head>
      <PageLoading />
      {contextHolder}
      <PricingManagementDrawer 
        isOpen={isManageDrawerOpen}
        showDrawer={setIsManageDrawerOpen}
        pricingData={selectedServer}
        triggerSuccess={() => setTriggerEffects(!triggerEffects)}
        setPricingData={setSelectedServer}
      />
      <main>
        <DashboardLayout title="Model Cost" loading={false} activemenu="adminpricing">
            <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
                <Breadcrumb
                    items={[
                    {
                        title:<Link href={"/dashboard"}>Dashboard</Link>,
                    },
                    {
                        title: 'Model Cost',
                    },
                    ]}
                />
            </Space>
            <Row gutter={[15, 15]}>
                <Col span={24}>
                <Card title="Pricing Management" 
                    extra={<Space>
                        <Button type="primary" onClick={() => {
                            setSelectedServer(null);
                            setIsManageDrawerOpen(true);
                        }}>Add Model Cost</Button>
                        </Space>} 
                >
                    <div className='table_container'>
                    <Table
                        dataSource={data}
                        rowKey="_id"
                        pagination={{ pageSize: 10 }}
                        columns={[
                            {
                                title: 'Model',
                                dataIndex: 'model',
                                key: 'model',
                            },
                            {
                                title: 'Input Cost',
                                dataIndex: 'input',
                                key: 'input',
                                render: (_, record) => `$${record?.pricing?.input || 0}`,
                            },
                            {
                                title: 'Output Cost',
                                dataIndex: 'output',
                                key: 'output',
                                render: (_, record) => `$${record?.pricing?.output || 0}`,
                            },
                            {
                                title: 'Cache Cost',
                                dataIndex: 'cache',
                                key: 'cache',
                                render: (_, record) => `$${record?.pricing?.cache || 0}`,
                            },

                            {
                                title: 'Actions',
                                key: 'actions',
                                render: (_, record) => (
                                    <Space size="middle">
                                        <Button size='small' onClick={() => {
                                            setSelectedServer(record);
                                            setIsManageDrawerOpen(true);
                                        }}><RiPencilFill /></Button>
                                        {/* <Button size='small' danger onClick={() => console.log('Delete', record._id)}><LuTrash2 /></Button> */}
                                    </Space>
                                ),
                            },
                        ]}
                        loading={loading}
                        bordered
                        size="small"
                    />
                    </div>
                </Card>
                </Col>
            </Row>
        </DashboardLayout>
      </main>
    </>
  )
}

export default withAdminAuth(PricingAdminPage);