import withAdminAuth from '@/components/auth/withAdminAuth';
import DashboardLayout from '@/components/layout/DashboardLayout';
import PageLoading from '@/components/utils/PageLoading';
import { Breadcrumb, Button, Card, Col, message, Row, Space, Table, Tag } from 'antd';
import Head from 'next/head';
import Link from 'next/link';
import React, { useState, useEffect } from 'react'
import { RiPencilFill } from "react-icons/ri";
import { LuTrash2 } from "react-icons/lu";
import ServerManagementDrawer from '@/drawers/ServersManagementDrwer';

const ServersAdminPage = () => {
    const [loading, setLoading] = useState(false);
    const [data, setData] = useState([]);
    const [serverCountList, setServerCountList] = useState([]);
    const [triggerEffects, setTriggerEffects] = useState(false);
    const [messageApi, contextHolder] = message.useMessage();
    const [isManageDrawerOpen, setIsManageDrawerOpen] = useState(false);
    const [selectedServer, setSelectedServer] = useState(null);

    useEffect(() => {
        const fetchServers = async () => {
            setLoading(true);
            try {
                const response = await fetch('/api/admin/servers/list', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                });
                const result = await response.json();
                if (result.status === 'success') {
                    setData(result.data);
                    setServerCountList(result.serverStatusCounts || []);
                } else {
                    messageApi.error(result.message || 'Failed to fetch servers');
                }
            } catch (error) {
                console.error('Error fetching servers:', error);
                messageApi.error('Failed to fetch servers');
            } finally {
                setLoading(false);
            }
        };

        fetchServers();
    }, [triggerEffects]);

    const getUsageCount = (serverUrl, status = 'total') => {
        const serverData = serverCountList.find(item => item.serverUrl === serverUrl);
        if (serverData) {
            switch (status) {
                case 'total':
                    return serverData.totalCount || 0;
                case 'scanned':
                    return serverData.scannedCount || 0;
                case 'active':
                    return serverData.activeCount || 0;
                case 'loading':
                    return serverData.loadingCount || 0;
                default:
                    return 0;
            }
        }
        return 0;
    };

    const handleCheckServerStatus = async (serverUrl) => {
        setLoading(true);
        try {
            const response = await fetch('/api/admin/servers/checkstatus', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ serverUrl }),
            });
            const result = await response.json();
            if (result.status === 'success') {
                messageApi.success(result.message || 'Server is active');
            } else {
                messageApi.error(result.message || 'Server is not responding');
            }
        } catch (error) {
            console.error('Error checking server status:', error);
            messageApi.error('Failed to check server status');
        } finally {
            setLoading(false);
        }
    };

  return (
    <>
      <Head>
        <title>Servers | Sendiee Admin</title>
        <meta name="viewport" content="width=device-width, initial-scale=1" />
      </Head>
      <PageLoading />
      {contextHolder}
      <ServerManagementDrawer 
        isOpen={isManageDrawerOpen}
        showDrawer={setIsManageDrawerOpen}
        serverData={selectedServer}
        triggerSuccess={() => setTriggerEffects(!triggerEffects)}
        setServerData={setSelectedServer}
      />
      <main>
        <DashboardLayout title="Servers" loading={false} activemenu="servers">
            <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
                <Breadcrumb
                    items={[
                    {
                        title:<Link href={"/dashboard"}>Dashboard</Link>,
                    },
                    {
                        title: 'Servers',
                    },
                    ]}
                />
            </Space>
            <Row gutter={[15, 15]}>
                <Col span={24}>
                <Card title="Server Management" 
                    extra={<Space>
                        <Button type="primary" onClick={() => {
                            setSelectedServer(null);
                            setIsManageDrawerOpen(true);
                        }}>Add Server</Button>
                        </Space>} 
                >
                    <div className='table_container'>
                    <Table
                        dataSource={data}
                        rowKey="_id"
                        pagination={{ pageSize: 10 }}
                        columns={[
                            {
                                title: 'Server',
                                dataIndex: 'serverUrl',
                                key: 'serverUrl',
                                render: (text) => (
                                    <Button type="link" onClick={() => handleCheckServerStatus(text)}>
                                        {text}
                                    </Button>
                                ),
                            },
                            {
                                title: 'Notes',
                                dataIndex: 'notes',
                                key: 'notes',
                            },
                            {
                                title: 'Status',
                                dataIndex: 'isActive',
                                key: 'isActive',
                                render: (text) => (
                                    <Tag color={text ? 'green' : 'yellow'}>
                                        {text ? 'Active' : 'Inactive'}
                                    </Tag>
                                ),
                            },
                            {
                                title: 'Organization',
                                dataIndex: 'orgId',
                                key: 'orgId',
                                render: (text) => text ? <Tag>{text.name}</Tag> : '-',
                            },
                            {
                                title: 'Usage',
                                dataIndex: 'maxLimits',
                                key: 'maxLimits',
                                render: (_, record) => <>
                                    {getUsageCount(record.serverUrl, 'total')} of {record.maxLimits}
                                </>,
                            },
                            {
                                title: 'Actions',
                                key: 'actions',
                                render: (_, record) => (
                                    <Space size="middle">
                                        <Button size='small' onClick={() => {
                                            setSelectedServer(record);
                                            setIsManageDrawerOpen(true);
                                        }}><RiPencilFill /></Button>
                                        {/* <Button size='small' danger onClick={() => console.log('Delete', record._id)}><LuTrash2 /></Button> */}
                                    </Space>
                                ),
                            },
                        ]}
                        loading={loading}
                        bordered
                        size="small"
                    />
                    </div>
                </Card>
                </Col>
            </Row>
        </DashboardLayout>
      </main>
    </>
  )
}

export default withAdminAuth(ServersAdminPage);