import withAdminAuth from '@/components/auth/withAdminAuth';
import DashboardLayout from '@/components/layout/DashboardLayout';
import PageLoading from '@/components/utils/PageLoading';
import { Avatar, Breadcrumb, Button, Card, Col, Flex, message, Row, Space, Table, Tag, Tooltip } from 'antd';
import Head from 'next/head';
import Link from 'next/link';
import React, { useState, useEffect } from 'react'
import { RiPencilFill } from "react-icons/ri";
import { useRouter } from 'next/router';
import { FaArrowRightLong } from "react-icons/fa6";
import { FaCrown } from "react-icons/fa6";
import ManageUsersDrawer from '@/drawers/ManageUsersDrawer';
import { FcGoogle } from "react-icons/fc";

const UsersAdminPage = () => {
    const [loading, setLoading] = useState(false);
    const [data, setData] = useState([]);
    const [triggerEffects, setTriggerEffects] = useState(false);
    const [messageApi, contextHolder] = message.useMessage();
    const [isDrawerOpen, setIsDrawerOpen] = useState(false);
    const [selectedUser, setSelectedUser] = useState(null);
    const router = useRouter();

    useEffect(() => {
        const fetchOrgs = async () => {
            setLoading(true);
            try {
                const response = await fetch('/api/admin/users/list');
                const result = await response.json();
                if (result.status === 'success') {
                    setData(result.data);
                } else {
                    messageApi.error(result.message || 'Failed to fetch users');
                }
            } catch (error) {
                console.error('Error fetching users:', error);
                messageApi.error('Failed to fetch users');
            } finally {
                setLoading(false);
            }
        };

        fetchOrgs();
    }, [triggerEffects]);

  return (
    <>
      <Head>
        <title>Users | Sendiee Admin</title>
        <meta name="viewport" content="width=device-width, initial-scale=1" />
      </Head>
      <PageLoading />
      {contextHolder}
      <ManageUsersDrawer isOpen={isDrawerOpen} showDrawer={setIsDrawerOpen} triggerSuccess={() => setTriggerEffects(!triggerEffects)} userData={selectedUser} setUserData={setSelectedUser} />
      <main>
        <DashboardLayout title="Users" loading={false} activemenu="adminusers">
            <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
                <Breadcrumb
                    items={[
                    {
                        title:<Link href={"/dashboard"}>Dashboard</Link>,
                    },
                    {
                        title: 'Users',
                    },
                    ]}
                />
            </Space>
            <Row gutter={[15, 15]}>
                <Col span={24}>
                <Card title="Users Management"
                    extra={
                        [
                            <Button key="add" type="primary" onClick={() => {
                                setSelectedUser(null);
                                setIsDrawerOpen(true);
                            }}>Add User</Button>
                        ]
                    }
                >
                    <div className='table_container'>
                    <Table
                        dataSource={data}
                        rowKey="_id"
                        pagination={{ pageSize: 10 }}
                        columns={[
                            {
                                title: 'Email',
                                dataIndex: 'email',
                                key: 'email',
                                render: (_, record) => (
                                    <Flex align='center' gap={5}>
                                        <Avatar src={ <img src={record.profilePicture || 'https://www.gravatar.com/avatar/00000000000000000000000000000000?d=mp&f=y'}></img>} />
                                        <span>{record.email}</span>
                                        {record.isSuperAdmin && <Tooltip title="Super Admin"><Tag icon={<FaCrown />} color="gold"></Tag></Tooltip>}
                                    </Flex>
                                )
                            },
                            {
                                title: 'Name',
                                dataIndex: 'name',
                                key: 'name',
                            },
                            {
                                title: 'Google Auth',
                                dataIndex: 'uid',
                                key: 'uid',
                                render: (text) => (<>{text && <Tag style={{
                                    display: 'inline-flex',
                                    alignItems: 'center',
                                    gap: '5px'
                                }}><FcGoogle /> Linked</Tag>}</>)
                            },
                            {
                                title: 'Password',
                                dataIndex: 'password',
                                key: 'password',
                                render: (text) => (<>{text ? 'Yes' : '-'}</>)
                            },
                            {
                                title: 'Status',
                                dataIndex: 'isActive',
                                key: 'isActive',
                                render: (text) => (
                                    <Tag color={text ? 'green' : 'yellow'}>
                                        {text ? 'Active' : 'Inactive'}
                                    </Tag>
                                ),
                            },
                            {
                                title: 'Created At',
                                dataIndex: 'createdAt',
                                key: 'createdAt',
                                render: (text) => new Date(text).toLocaleDateString(),
                            },
                            {
                                title: 'Last Login',
                                dataIndex: 'lastLogin',
                                key: 'lastLogin',
                                render: (text) => new Date(text).toLocaleDateString(),
                            },
                            {
                                title: 'Actions',
                                key: 'actions',
                                render: (_, record) => (
                                    <Space size="middle">
                                        <Button size='small' onClick={() => {
                                            setSelectedUser(record);
                                            setIsDrawerOpen(true);
                                        }}><RiPencilFill /></Button>
                                    </Space>
                                ),
                            },
                        ]}
                        loading={loading}
                        bordered
                        size="small"
                    />
                    </div>
                </Card>
                </Col>
            </Row>
        </DashboardLayout>
      </main>
    </>
  )
}

export default withAdminAuth(UsersAdminPage);