import dbConnect from '@/lib/db/dbConnect';
import { getServerSession } from 'next-auth';
import { withAdminAuth } from '../../withAdminAuth';
import { authOptions } from '../../auth/[...nextauth]';
import Server from '@/lib/models/Servers';
import QR from '@/lib/models/QR';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        // Fetch all servers
        const servers = await Server.find({})
        .populate('orgId', 'name')
        .sort({ createdAt: -1 });

        const serverStatusCounts = await QR.aggregate([
            {
                $group: {
                _id: "$serverUrl",
                totalCount: { $sum: 1 },
                scannedCount: {
                    $sum: { $cond: [{ $eq: ["$status", "scanned"] }, 1, 0] }
                },
                activeCount: {
                    $sum: { $cond: [{ $eq: ["$status", "active"] }, 1, 0] }
                },
                loadingCount: {
                    $sum: { $cond: [{ $eq: ["$status", "loading"] }, 1, 0] }
                }
                }
            },
            {
                $project: {
                serverUrl: "$_id",
                totalCount: 1,
                scannedCount: 1,
                activeCount: 1,
                loadingCount: 1,
                _id: 0
                }
            }
        ]);

        return res.status(200).json({
            status: 'success',
            message: 'Servers list fetched successfully',
            data: servers,
            serverStatusCounts: serverStatusCounts
        });
    }catch(err){
        console.log('Error fetching servers', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to fetch servers'
        })
    }
}

export default withAdminAuth(async(req, res) => {
  await endpoint(req, res);
});