import dbConnect from '@/lib/db/dbConnect';
import { getServerSession } from 'next-auth';
import { withAdminAuth } from '../../withAdminAuth';
import { authOptions } from '../../auth/[...nextauth]';
import Transaction from '@/lib/models/Transaction';
import Wallet from '@/lib/models/Wallet';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        let { orgId, amount } = req.body;
        
        if (!orgId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        // Add Funds to the wallet
        if (!amount || isNaN(amount) || amount <= 0) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Invalid amount provided'
            });
        }
        const wallet = await Wallet.findOneAndUpdate(
            { orgId: orgId },
            { $inc: { balance: (amount || 0) * 1000 } },
            { new: true, upsert: true }
        );
        if (!wallet) {
            return res.status(500).json({ 
                status: 'fail',
                message: 'Failed to update wallet balance'
            });
        }
        // Create a transaction record
        const transaction = new Transaction({
            orgId: orgId,
            amountConsumed: (amount || 0) * 1000,
            type: 'CREDIT',
            category: 'ADD_FUNDS',
        });

        await transaction.save();

        return res.status(200).json({
            status: 'success',
            message: 'Amount added successfully',
        });
    }catch(err){
        console.log('Error adding funds', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to add funds'
        })
    }
}

export default withAdminAuth(async(req, res) => {
  await endpoint(req, res);
});