import axios from 'axios';
import dbConnect from '@/lib/db/dbConnect'; // assumes dbConnect sets up mongoose
import BusinessCustomer from '@/lib/models/BusinessCustomer';

export default async function handler(req, res) {
  const { code, error_code, error_message } = req.query;
  console.log('Received query:', req.query);

  if (error_code || error_message) {
    console.error('Error during Meta OAuth:', error_code, error_message);
    return res.status(400).json({ error: 'Meta OAuth error', details: { error_code, error_message } });
  }

  if (!code) return res.status(400).json({ error: 'No code provided' });

  try {
    await dbConnect();

    const appId = process.env.NEXT_PUBLIC_META_APP_ID;
    const graphApiVersion = process.env.NEXT_PUBLIC_GRAPH_API_VERSION;
    const appSecret = process.env.META_APP_SECRET;

    // STEP 1: Exchange code for long-lived access token
    const tokenRes = await axios.get(`https://graph.facebook.com/${graphApiVersion}/oauth/access_token`, {
      params: {
        client_id: appId,
        client_secret: appSecret,
        redirect_uri: process.env.NEXT_PUBLIC_META_REDIRECT_URI,
        code,
      },
    });

    const accessToken = tokenRes.data.access_token;

    // STEP 2: Get user ID and businesses
    const meRes = await axios.get(`https://graph.facebook.com/${graphApiVersion}/me/accounts?access_token=${accessToken}`);
    console.log('User accounts:', meRes.data);
    const page = meRes.data.data[0]; // Assuming first page

    const businessesRes = await axios.get(`https://graph.facebook.com/${graphApiVersion}/me/businesses?access_token=${accessToken}`);
    const businessId = businessesRes.data.data[0]?.id;

    console.log('Business account:', businessesRes.data);
    if (!businessId) throw new Error("No business accounts associated with this user.");

    // STEP 3: Get WABA (WhatsApp Business Account)
    const wabaRes = await axios.get(`https://graph.facebook.com/${graphApiVersion}/${businessId}/owned_whatsapp_business_accounts?access_token=${accessToken}`);
    console.log('WABA:', wabaRes.data);
    const wabaId = wabaRes.data.data[0]?.id;

    // STEP 4: Get phone numbers
    const phoneRes = await axios.get(`https://graph.facebook.com/${graphApiVersion}/${wabaId}/phone_numbers?access_token=${accessToken}`);
    console.log('Phone numbers:', phoneRes.data);
    const phoneNumberId = phoneRes.data.data[0]?.id;

    // Save in DB
    const customer = await BusinessCustomer.create({
      userId: page.id,
      businessName: page.name,
      accessToken,
      wabaId,
      phoneNumberId,
      businessId,
    });

    return res.redirect(`/onboard/success?id=${customer._id}`);
  } catch (err) {
    console.error('Facebook auth error:', err.response?.data || err.message);
    return res.status(500).json({ error: 'Onboarding failed' });
  }
}
