import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import Server from '@/lib/models/Servers';
import axios from 'axios';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId } = req.body;
        if (!orgId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameter is missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        // Fetch server URL from the database
        // First check if the org has a server URL, if not, use the default server URL
        let server = await Server.findOne({
            orgId: orgId,
            isActive: true,
        });

        if (!server) {
            // If no server found for the org, use the default server
            server = await Server.findOne({ type: 'default', isActive: true });
        }

        if (!server) {
            return res.status(404).json({
                status: 'fail',
                message: 'No active server found'
            });
        }
        const serverUrl = server.serverUrl;

        // Post to the server to create a WhatsApp instance
        const headers = {
            'Content-Type': 'application/json',
            'x-api-key': `${process.env.SERVER_API_KEY}`
        };
        const response = await axios.post(`${serverUrl}/add`, {
            orgId: orgId,
            userId: userId,
            serverUrl: serverUrl
        }, {
            headers: headers
        });

        if (response.status !== 200) {
            return res.status(response.status).json({
                status: 'fail',
                message: response.data.message || 'Failed to create WhatsApp instance'
            });
        }

        return res.status(200).json({
            status: 'success',
            message: 'WhatsApp Instance created successfully',
            data: response.data
        });
    }catch(err){
        console.log('Error creating whatsapp instance:', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to create whatsapp instance'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'admin', 'limited'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});