import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import QR from '@/lib/models/QR';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId } = req.body;
        const { type } = req.query;
        if (!orgId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }
        
        let qrData;

        if(type && type === 'active') {
            // Fetch only active WhatsApp instances
            qrData = await QR.find({
                orgId: orgId,
                status: 'scanned'
            }).sort({ createdAt: -1 });
        } else{
            // Fetch all WhatsApp instances for the organization
            qrData = await QR.find({
                orgId: orgId
            }).sort({ createdAt: -1 });
        }


        return res.status(200).json({
            status: 'success',
            message: 'WhatsApp Instances fetched successfully',
            data: qrData
        });
    }catch(err){
        console.log('Error fetching instances', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to fetch WhatsApp Instances'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'read', 'admin', 'limited'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});