import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import QR from '@/lib/models/QR';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, instanceId } = req.body;
        if (!orgId || !instanceId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        const qrData = await QR.findOne({
            instance_id: instanceId,
            orgId: orgId,
        });

        if (!qrData) {
            return res.status(404).json({
                status: 'fail',
                message: 'WhatsApp Instance not found'
            });
        }

        return res.status(200).json({
            status: 'success',
            message: 'WhatsApp Instance QR status fetched successfully',
            data: qrData
        });
    }catch(err){
        console.log('Error fetching qr status', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to fetch WhatsApp Instance QR status'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'admin', 'limited'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});