import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import { nanoid } from 'nanoid';
import AIModel from '@/lib/models/AIModel';
import QR from '@/lib/models/QR';
import axios from 'axios';
import Server from '@/lib/models/Servers';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, name, description, instance_id, provider, model, systemPrompt, maxTokens, temperature, topP, tools, voiceEnabled, imageEnabled } = req.body;
        if (!orgId || !name || !provider || !model) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameter is missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        let phone = '';
        if (instance_id) {

            // Fetch the instance details from QR model
            const qrData = await QR.findOne({
                orgId: orgId,
                _id: instance_id});
            phone = qrData?.phone || '';
        }

        // Store the AI model in the database
        const modelData = {
            orgId,
            instance_id,
            phone,
            name,
            description: description || '',
            modelId: nanoid(14), // Unique model ID
            provider,
            model,
            maxTokens: maxTokens || 250,
            temperature: temperature,
            topP: topP,
            systemPrompt: systemPrompt || '',
            tools: tools || [],
            voiceEnabled: voiceEnabled || false,
            imageEnabled: imageEnabled || false,
            isActive: true, // Default to active
        };

        const newModel = await AIModel.create(modelData);

        // Reset redis cache in server [Temporary solution]

        if(phone && phone !== '') {
            // Fetch server URL from the database
            // First check if the org has a server URL, if not, use the default server URL
            let server = await Server.find({
                isActive: true,
            });

            if(server.length){
                for(const s of server){
                    const headers = {
                        'Content-Type': 'application/json',
                        'x-api-key': `${process.env.SERVER_API_KEY}`
                    };
                    axios.post(`${s.serverUrl}/reset`, {
                                orgId: orgId,
                                userId: userId,
                                phone: phone,
                            }, {
                                headers: headers
                    }).catch(err => {
                        console.error('Error resetting server cache for:', phone, err);
                    });
                }
            }
        }

        return res.status(200).json({
            status: 'success',
            message: 'Model created successfully',
            data: newModel
        });
    }catch(err){
        console.log('Error creating model:', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to create AI model'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'admin'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});