import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import AIModel from '@/lib/models/AIModel';
import Server from '@/lib/models/Servers';
import axios from 'axios';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, modelId, phone } = req.body;
        
        if (!orgId || !modelId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        // Delete the AI model from the database
        const deletedModel = await AIModel.findOneAndDelete({ orgId, modelId: modelId });

        if (!deletedModel) {
            return res.status(404).json({
                status: 'fail',
                message: 'Model not found'
            });
        }

        if(phone && phone !== '') {
            // Fetch server URL from the database
            // First check if the org has a server URL, if not, use the default server URL
            let server = await Server.find({
                isActive: true,
            });
    
            if(server.length){
                for(const s of server){
                    const headers = {
                        'Content-Type': 'application/json',
                        'x-api-key': `${process.env.SERVER_API_KEY}`
                    };
                    axios.post(`${s.serverUrl}/reset`, {
                                orgId: orgId,
                                userId: userId,
                                phone: phone,
                            }, {
                                headers: headers
                    })
                    .catch(err => {
                        console.error('Error resetting server cache for:', phone, err);
                    });
                }
            }
        }


        return res.status(200).json({
            status: 'success',
            message: 'Model deleted successfully',
        });
    }catch(err){
        console.log('Error deleting AI model', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to delete AI model'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'admin'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});