import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import AIModel from '@/lib/models/AIModel';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId } = req.body;
        
        if (!orgId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        const modelData = await AIModel.find({
            orgId: orgId
        })
        .populate({
            path: 'instance_id',
            select: 'phone', // select only phone field from QR
        })
        .sort({ createdAt: -1 });

        return res.status(200).json({
            status: 'success',
            message: 'AI Models fetched successfully',
            data: modelData
        });
    }catch(err){
        console.log('Error fetching AI models', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to fetch AI models'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'read', 'admin', 'limited'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});