import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import AIModel from '@/lib/models/AIModel';
import QR from '@/lib/models/QR';
import Server from '@/lib/models/Servers';
import axios from 'axios';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, modelId, name, description, 
            instance_id, provider, model, systemPrompt, 
            maxTokens, temperature, topP, tools, voiceEnabled, 
            imageEnabled, isActive, phoneNumbersList, isNumbersExcluded,
            enableCommand, disableCommand, disableDuration
         } = req.body;
        if (!orgId || !name || !provider || !model || !modelId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameter is missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        let phone = '';
        if (instance_id) {

            // Fetch the instance details from QR model
            const qrData = await QR.findOne({
                orgId: orgId,
                _id: instance_id});
            phone = qrData?.phone || '';
        }

        // Store the AI model in the database
        const modelData = {
            instance_id,
            ...((phone && phone != '') ? { phone } : {}),
            name,
            description: description || '',
            provider,
            model,
            maxTokens: maxTokens || 250,
            temperature: temperature,
            topP: topP,
            systemPrompt: systemPrompt || '',
            tools: tools || [],
            voiceEnabled: voiceEnabled || false,
            imageEnabled: imageEnabled || false,
            isActive: isActive,
            phoneNumbersList: phoneNumbersList || [],
            isNumbersExcluded: isNumbersExcluded || false,
            enableCommand: enableCommand || '',
            disableCommand: disableCommand || '',
            disableDuration: disableDuration || 0,
        };

        // Update the model in the database
        const updatedModel = await AIModel.findOneAndUpdate(
            { orgId, modelId },
            modelData,
            { new: true, runValidators: true }
        );

        // Reset redis cache in server [Temporary solution]

        if(phone && phone !== '') {
            // Fetch server URL from the database
            // First check if the org has a server URL, if not, use the default server URL
            let server = await Server.find({
                isActive: true,
            });

            if(server.length){
                for(const s of server){
                    const headers = {
                        'Content-Type': 'application/json',
                        'x-api-key': `${process.env.SERVER_API_KEY}`
                    };
                    axios.post(`${s.serverUrl}/reset`, {
                                orgId: orgId,
                                userId: userId,
                                phone: phone,
                            }, {
                                headers: headers
                    })
                    .catch(err => {
                        console.error('Error resetting server cache for:', phone, err);
                    });
                }
            }
        }

        return res.status(200).json({
            status: 'success',
            message: 'Model updated successfully',
        });
    }catch(err){
        console.log('Error updating model:', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to update AI model'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'admin', 'limited'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});