import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import Organization from '@/lib/models/Organization';
import User from '@/lib/models/User';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, email, access, orgOwnerId } = req.body;
        
        if (!orgId || !email) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed'
            });
        }

        // Check if the user exists
        const user = await User.findOne({ email: email.trim().toLowerCase() });
        let userData = null;
        if (!user) {
            // Create a new user if they don't exist
            const newUser = new User({
                email: email.trim().toLowerCase(),
                name: email.split('@')[0], // Default name from email
            });
            await newUser.save();
            userData = newUser;
        } else{
            // If user already exists, ensure they are not already a member of the organization
            const isMember = await Organization.findOne({
                _id: orgId,
                'members.userId': user._id
            });

            if (isMember) {
                return res.status(400).json({ 
                    status: 'fail',
                    message: 'User is already a member of this organization'
                });
            }
            // If user an owner of this organization
            if (user._id.toString() === orgOwnerId) {
                return res.status(400).json({ 
                    status: 'fail',
                    message: 'You cannot add the organization owner as a member'
                });
            }

            userData = user;
        }
        // Add the user to the organization
        await Organization.updateOne(
            { _id: orgId },
            { $addToSet: { members: { userId: userData._id, access: access || 'read' } } }
        );

        return res.status(200).json({
            status: 'success',
            message: 'Member added successfully',
        });
    }catch(err){
        console.log('Error adding member', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to add member'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'admin'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});