import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import Organization from '@/lib/models/Organization';
import User from '@/lib/models/User';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, memberId, access } = req.body;
        
        if (!orgId || !memberId || !access) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed'
            });
        }

        // Update user's access in organization
        const org = await Organization.findOneAndUpdate(
            { _id: orgId, 'members.userId': memberId },
            { $set: { 'members.$.access': access } },
            { new: true }
        );

        if (!org) {
            return res.status(404).json({ 
                status: 'fail',
                message: 'Organization not found or member not part of this organization'
            });
        }

        return res.status(200).json({
            status: 'success',
            message: 'Member access updated successfully',
        });
    }catch(err){
        console.log('Error updating member', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to update member'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'admin'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});