import dbConnect from '@/lib/db/dbConnect';
import User from '@/lib/models/User';
import { withAuth } from '../withAuth';
import { nanoid } from 'nanoid';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import Organization from '@/lib/models/Organization';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { name, description, orgId } = req.body;
        if (!name || !orgId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to create an organization' 
            });
        }

        const organization = {
            name: name,
            description: description || '',
        }
        const orgData = await Organization.findOneAndUpdate(
            { _id: orgId },
            organization,
            { new: true, runValidators: true }
        );
        if (!orgData) {
            return res.status(404).json({ 
                status: 'fail',
                message: 'Organization not found' 
            });
        }
        return res.status(201).json({
            status: 'success',
            message: 'Organization updated successfully',
        });
    }catch(err){
        console.log('Error updating organization:', err);
        if (err.code === 11000) {
            return res.status(400).json({
                status: 'fail',
                message: 'Organization with this name already exists'
            });
        }
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to update organization'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'admin'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});