import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import { nanoid } from 'nanoid';
import AIModel from '@/lib/models/AIModel';
import AITool from '@/lib/models/AITool';
import Server from '@/lib/models/Servers';
import axios from 'axios';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, name, async, parameters, serverUrl, strict, isStatic, response, toolType, description, secretToken  } = req.body;
        if (!orgId || !name || !parameters || !toolType) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameter is missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        // Store the AI Tool in the database
        const toolData = {
            orgId,
            name,
            description: description || '',
            toolId: nanoid(14), // Unique tool ID
            async: async || false,
            parameters: parameters || {},
            serverUrl,
            isStatic: isStatic || false,
            response: response || '',
            secretToken: secretToken || '',
            isActive: true, // Default to active
            strict: strict || false,
            toolType: 'function', // Default to 'function' 
        };

        const newTool = await AITool.create(toolData);

        // Reset redis cache in server [Temporary solution]

        // Fetch server URL from the database
        // First check if the org has a server URL, if not, use the default server URL
        let server = await Server.find({
            isActive: true,
        });

        if(server.length){
            for(const s of server){
                const headers = {
                    'Content-Type': 'application/json',
                    'x-api-key': `${process.env.SERVER_API_KEY}`
                };
                axios.post(`${s.serverUrl}/reset`, {
                            orgId: orgId,
                            userId: userId,
                        }, {
                            headers: headers
                });
            }
        }

        return res.status(200).json({
            status: 'success',
            message: 'Tool created successfully',
            data: newTool
        });
    }catch(err){
        console.log('Error creating tool:', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to create AI Tool'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'admin', 'limited'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});