import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import AIModel from '@/lib/models/AIModel';
import AITool from '@/lib/models/AITool';
import Server from '@/lib/models/Servers';
import axios from 'axios';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, _id } = req.body;
        
        if (!orgId || !_id) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        // Delete the AI tool from the database
        const deletedTool = await AITool.findOneAndDelete({ orgId, _id: _id });

        
        if (!deletedTool) {
            return res.status(404).json({
                status: 'fail',
                message: 'Tool not found'
            });
        }
        
        // Delete the tool from the AI model if it exists
        await AIModel.updateMany(
            { orgId: orgId, tools: _id },
            { $pull: { tools: _id } }
        );

        return res.status(200).json({
            status: 'success',
            message: 'Tool deleted successfully',
        });
    }catch(err){
        console.log('Error deleting AI tool', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to delete AI tool'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'admin'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});