import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import AITool from '@/lib/models/AITool';
import AIModel from '@/lib/models/AIModel';
import Server from '@/lib/models/Servers';
import axios from 'axios';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        const { orgId, toolId, isActive, _id } = req.body;
        
        if (!orgId || !toolId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        // Update the AI tool status in the database
        await AITool.findOneAndUpdate(
            { orgId, toolId },
            { isActive: isActive },
            { new: true }
        );

        // Reset redis cache in server [Temporary solution]
        // Get the models associated with the tool
        const models = await AIModel.find({ orgId, tools: _id });

        if (models.length > 0) {
            let activeServers = await Server.find({
                isActive: true,
            });

            // Get distinct phone numbers from models
            const distinctPhones = [...new Set(
                models
                    .filter(model => model.phone && model.phone !== '' && model.isActive)
                    .map(model => model.phone)
            )];

            if (distinctPhones.length > 0 && activeServers.length) {
                activeServers.forEach(server => {
                    const headers = {
                        'Content-Type': 'application/json',
                        'x-api-key': process.env.SERVER_API_KEY
                    };

                    distinctPhones.forEach(phone => {
                        axios.post(`${server.serverUrl}/reset`, {
                            orgId,
                            userId,
                            phone
                        }, { headers })
                        .catch(err => {
                            console.error(`Failed reset for ${phone} on ${server.serverUrl}:`, err.message);
                        });
                    });
                });
            }
        }

        return res.status(200).json({
            status: 'success',
            message: 'Tool status updated successfully',
        });
    }catch(err){
        console.log('Error updating AI tool', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to update AI tool status'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'admin', 'limited'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});