/**
 * API Route: /api/users/getaccess
 * Date: June 2025
 *
 * Description:
 *   Handles POST requests to fetch access details for a user.
 *   Returns user permissions and roles.
 *   Requires authentication.
 */
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import dbConnect from '@/lib/db/dbConnect';
import User from '@/lib/models/User';
import Organization from '@/lib/models/Organization';

export default withAuth(async (req, res) => {
    if(req.method == 'POST'){
        try {
            const session = await getServerSession(req, res, authOptions)
    
            const userId = session?.user?.userId;
            if(userId === undefined || userId === null || userId === '') {
                res.status(400).json({ status: 'fail', message: 'Please login to get user access' });
                return;
            }

            await dbConnect();

            console.log('Fetching user access for userId:', userId);
            
            // Fetch user details
            const user = await User.findById(userId).select('email name profilePicture isActive preferences isSuperAdmin');
            if (!user) {
                res.status(404).json({ status: 'fail', message: 'User not found' });
                return;
            }

            // Fetch all organizations the user is a member and owner of
            const organizationsRaw = await Organization.find({
                $or: [
                    { members: { $elemMatch: { userId: userId, isActive: true } } },
                    { ownerId: userId }
                ]
            }).select('name uniqueId description logoUrl isActive members ownerId');

            // Add isOwner field to each organization
            const organizations = organizationsRaw.map(org => ({
                ...org.toObject(),
                isOwner: org.ownerId?.toString() === userId.toString()
            }));

            res.status(200).json({ 
                status: 'success', 
                message: 'User access retrieved successfully', 
                result: {
                    user: {
                        email: user.email,
                        name: user.name,
                        profilePicture: user.profilePicture,
                        isActive: user.isActive,
                        preferences: user.preferences,
                        isSuperAdmin: user.isSuperAdmin,
                    },
                    organizations: organizations
                }
            });
        } catch (err) {
          console.error('Error executing getaccess query: ', err);
          res.status(500).json({ status : 'fail', message: 'Failed to get user access' });
        } 
    }else{
        res.status(401 ).json({ status : 'fail', message: 'Not allowed' });
    }
})
