import dbConnect from '@/lib/db/dbConnect';
import { withAuth } from '../withAuth';
import { getServerSession } from 'next-auth';
import { authOptions } from '../auth/[...nextauth]';
import { checkOrgPermissions } from '@/middlewares/checkorgaccess';
import Transaction from '@/lib/models/Transaction';

const endpoint = async (req, res) => {
    if (req.method !== 'POST') {
        return res.status(405).json({ 
        status: 'fail',
        message: 'Method not allowed' 
        });
    }
    await dbConnect();
    try{
        let { orgId, filterType, dateRange } = req.body;
        
        if (!orgId) {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Required parameters are missing'
            });
        }

        const session = await getServerSession(req, res, authOptions)

        const userId = session?.user?.userId;

        if (!userId || userId === undefined || userId === null || userId === '') {
            return res.status(400).json({ 
                status: 'fail',
                message: 'Please login to proceed' 
            });
        }

        // Fetch the wallet logs for the organization
        let logs = [];

        // Build query object
        const query = { orgId: orgId };
        if (filterType !== '' && filterType !== 'all' && filterType !== undefined && filterType !== null) {
            query.type = filterType === 'credit' ? 'CREDIT' : 'DEBIT';
        }

        if (dateRange && Array.isArray(dateRange) && dateRange.length === 2 && dateRange[0] && dateRange[1]) {
            dateRange = dateRange.map(date => {
                const parts = date.split('T');
                return parts[0];
            });
        }

        if (
            dateRange &&
            Array.isArray(dateRange) &&
            dateRange.length === 2 &&
            dateRange[0] &&
            dateRange[1]
        ) {
            // Assume dateRange[0] and dateRange[1] are in IST (UTC+5:30)
            // Convert to UTC
            const istToUtc = (dateStr, isEndOfDay) => {
                const date = new Date(dateStr);
                // Subtract 5 hours 30 minutes
                if (isEndOfDay) {
                    date.setHours(23, 59, 59, 999);
                } else {
                    date.setHours(0, 0, 0, 0);
                }
                date.setMinutes(date.getMinutes() - 330);
                return date;
            };
            query.timestamp = {
                $gte: istToUtc(dateRange[0], false),
                $lte: istToUtc(dateRange[1], true),
            };
        }

        logs = await Transaction.find(query).sort({ createdAt: -1 });

        return res.status(200).json({
            status: 'success',
            message: 'Wallet logs fetched successfully',
            logs: logs
        });
    }catch(err){
        console.log('Error fetching wallet logs', err);
        return res.status(500).json({
            status: 'fail',
            message: err.message || 'Failed to fetch wallet logs'
        })
    }
}

export default withAuth(async(req, res) => {
  const permissionsRequired = ['owner', 'write', 'read', 'admin', 'limited'];
  const orgId = req.query.orgId || req.body.orgId || req.headers['x-org-id'] || '';

  await checkOrgPermissions(permissionsRequired, orgId)(req, res, endpoint);
});