import checkPagePermissions from '@/components/auth/checkPagePermissions'
import withAuth from '@/components/auth/withAuth'
import DashboardLayout from '@/components/layout/DashboardLayout'
import PageLoading from '@/components/utils/PageLoading'
import { AI_PROVIDERS, OPENAI_MODELS } from '@/constant/variables'
import AppContext from '@/context/AppContext'
import { Breadcrumb, Space, Row, Col, message, Card, Tag, Form, Input, InputNumber, Typography, Select, Slider, AutoComplete, Alert, Button, Flex, Switch } from 'antd'
import Head from 'next/head'
import Link from 'next/link'
import { useRouter } from 'next/router'
import React, { useState, useEffect, useContext } from 'react'

const { Title, Text } = Typography;

const CreateModelPage = () => {
    const [loading, setLoading] = useState(false);
    const router = useRouter();
    const context = useContext(AppContext);
    const [messageApi, contextHolder] = message.useMessage();
    const [instances, setInstances] = useState([]);
    const [tools, setTools] = useState([]);
    const [form] = Form.useForm();
    const [formValues, setFormValues] = useState({
        name: '',
        description: '',
        instance_id: null,
        provider: 'openai',
        model: 'gpt-4.1',
        systemPrompt: '',
        maxTokens: 250,
        temperature: 0.7,
        topP: 1,
        isActive: true,
        tools: [],
    });

    useEffect(() => {
        const fetchInstances = async () => {
            try {
                setLoading(true);
                const response = await fetch('/api/instance/list?type=active', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        orgId: context?.activeOrg?._id,
                    }),
                });
                const data = await response.json();
                if (data.status === 'success') {
                    setInstances(data.data);
                } else {
                    messageApi.open({
                        type: 'error',
                        content: data.message || 'Failed to fetch instances',
                    });
                }
            } catch (error) {
                console.error('Error fetching instances:', error);
                messageApi.open({
                    type: 'error',
                    content: 'Failed to fetch instances',
                });
            } finally {
                setLoading(false);
            }
        }
        const fetchTools = async () => {
            try {
                setLoading(true);
                const response = await fetch('/api/tools/list?type=active', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        orgId: context?.activeOrg?._id,
                    }),
                });
                const data = await response.json();
                if (data.status === 'success') {
                    setTools(data.data);
                } else {
                    messageApi.open({
                        type: 'error',
                        content: data.message || 'Failed to fetch tools',
                    });
                }
            } catch (error) {
                console.error('Error fetching tools:', error);
                messageApi.open({
                    type: 'error',
                    content: 'Failed to fetch tools',
                });
            } finally {
                setLoading(false);
            }
        }
        fetchInstances();
        fetchTools();
    }, [context?.activeOrg?._id]);
    
    const handleValueChange = (name, value) => {
        setFormValues(prev => ({
            ...prev,
            [name]: value,
        }));
    }

    const handleSubmit = async (values) => {
        setLoading(true);
        try {
            const res = await fetch('/api/models/create', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    ...values,
                    orgId: context?.activeOrg?._id,
                }),
            });
            const data = await res.json();
            if (data.status === 'success') {
                messageApi.success('AI Model created successfully!');
                router.push('/dashboard/models');
            } else {
                messageApi.error(data.message || 'Failed to create model');
            }
        } catch (err) {
            console.error('Error creating model:', err);
            messageApi.error('Error creating model');
        } finally {
            setLoading(false);
        }
    }

    const providerModelMap = {
        openai: OPENAI_MODELS,
    };
    
  return (
    <>
    <Head>
        <title>Create AI Model | Sendiee Dashboard</title>
        <meta name="description" content="Sendiee dashboard" />
        <meta name="viewport" content="width=device-width, initial-scale=1" />
        <link rel="icon" href="/favicon.ico" />
    </Head>
    {contextHolder}
    <PageLoading />
    <DashboardLayout title="Create AI Model" loading={false} activemenu="models">
    <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
        <Breadcrumb
            items={[
            {
                title:<Link href={"/dashboard"}>Dashboard</Link>,
            },
            {
                title:<Link href={"/dashboard/models"}>Models</Link>,
            },
            {
                title: 'Create Model',
            },
            ]}
        />
    </Space>

    <Card title="Create AI Model" style={{ maxWidth: 700, margin: '0 auto' }}>
      <Form
        form={form}
        layout="vertical"
        onFinish={handleSubmit}
        initialValues={formValues}
        onValuesChange={(changedValues) => {
          const [name, value] = Object.entries(changedValues)[0];
          handleValueChange(name, value);
        }}
        autoComplete="off"
      >
        <Row gutter={[10, 10]}>
            <Col span={24}>
                <Form.Item
                name="name"
                label="Model Name"
                rules={[{ required: true, message: 'Please enter model name' }]}
                >
                <Input placeholder="Enter model name" showCount maxLength={60} size="large" />
                </Form.Item>
            </Col>
            <Col span={24}>
                <Form.Item
                name="description"
                label="Description"
                >
                <Input.TextArea rows={3} placeholder="Describe this model (optional)" size='large' />
                </Form.Item>
            </Col>
            <Col xs={24} lg={12}>
                <Form.Item
                name="instance_id"
                label="WhatsApp Account"
                // rules={[{ required: true, message: 'Please select WhatsApp account' }]}
                >
                <Select
                    placeholder="Select instance"
                    loading={loading}
                    options={instances.map(inst => ({
                        label: inst.phone || inst.name,
                        value: inst._id,
                    }))}
                    showSearch
                    size='large'
                    optionFilterProp="label"
                />
                </Form.Item>
            </Col>
            <Col xs={24} lg={12}>
                <Form.Item
                name="provider"
                label="Provider"
                rules={[{ required: true, message: 'Please select provider' }]}
                >
                <Select disabled defaultValue="openai" size='large'>
                    {AI_PROVIDERS.map(provider => (
                        <Select.Option key={provider.value} value={provider.value}>
                            {provider.label}
                        </Select.Option>
                    ))}
                </Select>
                </Form.Item>
            </Col>
            <Col xs={24} lg={24}>
                <Form.Item
                name="model"
                label="Model"
                rules={[{ required: true, message: 'Please select model' }]}
                >
                <Select 
                    placeholder="Select model"
                    options={providerModelMap[formValues.provider] || []}
                    value={formValues.model}
                    onChange={(value) => handleValueChange('model', value)}
                    size='large'
                    optionFilterProp="label"
                    showSearch
                >
                </Select>
                </Form.Item>
            </Col>
            <Col xs={24} lg={24}>
                <Form.Item
                name="tools"
                label="Tools"
                >
                <Select
                    mode="multiple"
                    placeholder="Select tools"
                    value={formValues.tools}
                    onChange={(value) => handleValueChange('tools', value)}
                    size='large'
                    style={{ width: '100%' }}
                    options={tools.map(tool => ({
                        label: tool.name,
                        value: tool._id,
                    }))}
                    optionFilterProp="label"
                    showSearch
                />
                </Form.Item>
            </Col>
            <Col xs={24} lg={24}>
                <Form.Item
                name="systemPrompt"
                label="System Prompt"
                >
                <Input.TextArea size='large' rows={10} placeholder="System prompt" 
                />
                </Form.Item>
            </Col>
            <Col xs={24} lg={12}>
                <Form.Item
                name="maxTokens"
                label="Max Tokens"
                rules={[{ required: true, message: 'Please enter max tokens' }]}
                >
                <InputNumber size='large' min={1} max={4096} style={{ width: '100%' }} />
                </Form.Item>
            </Col>
            <Col xs={24} lg={12}>
                <Form.Item
                name="topP"
                label="Top P"
                tooltip="Controls diversity via nucleus sampling"
                >
                <InputNumber size='large' min={0} max={1} step={0.01} style={{ width: '100%' }} />
                </Form.Item>
            </Col>
            <Col xs={24} lg={24}>
                <Form.Item
                name="temperature"
                label="Temperature"
                tooltip="Controls randomness: 0 (deterministic) to 1 (most random)"
                >
                <Slider size='large' min={0} max={1} step={0.01} />
                </Form.Item>
            </Col>
            <Col span={24}>
                <Card size='small' style={{ marginBottom: '15px' }}>  
                    <Flex justify="space-between" align="center">
                        <Flex vertical>
                            <Title level={4}>Voice Enabled</Title>
                            <Text>Enable AI to respond to voice messages.</Text>
                        </Flex>
                        <Form.Item
                            name="voiceEnabled"
                        >
                            <Switch />
                        </Form.Item>
                    </Flex>
                </Card>
            </Col>
            <Col span={24}>
                <Card size='small' style={{ marginBottom: '15px' }}>  
                    <Flex justify="space-between" align="center">
                        <Flex vertical>
                            <Title level={4}>Image Enabled</Title>
                            <Text>Enable AI to recognise and respond to images.</Text>
                        </Flex>
                        <Form.Item
                            name="imageEnabled"
                        >
                            <Switch />
                        </Form.Item>
                    </Flex>
                </Card>
            </Col>
            <Col span={24}>
                <Form.Item>
                <Button type="primary" size='large' htmlType="submit" loading={loading} block>
                    Create Model
                </Button>
                </Form.Item>
            </Col>
        </Row>
      </Form>
    </Card>
    </DashboardLayout>
    </>
  )
}

export default withAuth(checkPagePermissions(['owner', 'admin', 'write'])(CreateModelPage));