import checkPagePermissions from '@/components/auth/checkPagePermissions'
import withAuth from '@/components/auth/withAuth'
import DashboardLayout from '@/components/layout/DashboardLayout'
import PageLoading from '@/components/utils/PageLoading'
import AppContext from '@/context/AppContext'
import { Breadcrumb, Button, Card, Col, message, Popconfirm, Row, Space, Table, Tag, Switch } from 'antd'
import Head from 'next/head'
import Link from 'next/link'
import { useRouter } from 'next/router'
import React, { useContext, useEffect, useState } from 'react'

const ModelsPage = () => {
  const [loading, setLoading] = useState(false);
  const [data, setData] = useState([]);
  const [triggerEffects, setTriggerEffects] = useState(false);
  const [messageApi, contextHolder] = message.useMessage();
  const context = useContext(AppContext);
  const router = useRouter();
  const orgId = context?.activeOrg?._id || null;

  const fetchModels = async () => {
    setLoading(true);
    try {
      const response = await fetch('/api/models/list', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ orgId }),
      });
      const result = await response.json();
      if (result.status === 'success') {
        setData(result.data);
      } else {
        messageApi.error(result.message || 'Failed to fetch AI models');
      }
    } catch (error) {
      console.error('Error fetching AI models:', error);
      messageApi.error('Failed to fetch AI models');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (orgId) {
      fetchModels();
    } else {
      messageApi.error('Organization ID is required to fetch models');
      // router.push('/dashboard');
    }
  }, [orgId, triggerEffects]);

  const handleToggleModelStatus = async (modelId, isActive, phone) => {
    setLoading(true);
    try {
      const response = await fetch('/api/models/toggle', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ modelId, isActive, orgId, phone }),
      });
      const result = await response.json();
      if (result.status === 'success') {
        messageApi.success(result.message || 'Model status updated successfully');
        setTriggerEffects(!triggerEffects);
      } else {
        messageApi.error(result.message || 'Failed to update model status');
      } 
    } catch (error) {
      console.error('Error updating model status:', error);
      messageApi.error('Failed to update model status');
    } finally {
      setLoading(false);
    }
  }

  return (
    <>
        <Head>
            <title>AI Models | Sendiee Dashboard</title>
            <meta name="description" content="Sendiee dashboard" />
            <meta name="viewport" content="width=device-width, initial-scale=1" />
            <link rel="icon" href="/favicon.ico" />
        </Head>
        <PageLoading />
        <main>
            <DashboardLayout title="Models" loading={false} activemenu="models">
              <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
                    <Breadcrumb
                        items={[
                        {
                            title:<Link href={"/dashboard"}>Dashboard</Link>,
                        },
                        {
                            title: 'Models',
                        },
                        ]}
                    />
              </Space>
              <Row gutter={[15, 15]}>
                  <Col span={24}>
                      <Card title="AI Models" extra={
                          <Space>
                              <Button type='primary' onClick={() => {router.push('/dashboard/models/create')}}>Add Model</Button>
                          </Space>
                      }
                      style={{
                          width: '100%',
                      }}
                      >
                        <div className='table_container'>
                          <Table 
                              bordered={true}
                              loading={loading}
                              rowKey="modelId"
                              size='small'
                              columns={[
                                  {
                                      title: 'Status',
                                      dataIndex: 'isActive',
                                      key: 'isActive',
                                      render: (text, record) => (
                                          <Switch 
                                              checked={text}
                                              onChange={(checked) => handleToggleModelStatus(record.modelId, checked, record.phone )}
                                              loading={loading}
                                          />
                                      ),
                                  },
                                  {
                                      title: 'Model Name',
                                      dataIndex: 'name',
                                      key: 'name',
                                  },
                                  {
                                      title: 'WhatsApp',
                                      dataIndex: 'phone',
                                      key: 'phone',
                                      render: (text, record) => (
                                        <>{record?.instance_id?.phone || <Tag color='red'>Not Linked</Tag>}</>
                                      )
                                  },
                                  {
                                      title: 'Provider',
                                      dataIndex: 'provider',
                                      key: 'provider',
                                  },
                                  {
                                      title: 'Phone',
                                      dataIndex: 'phone',
                                      key: 'phone',
                                  },
                                  {
                                      title: 'Model',
                                      dataIndex: 'model',
                                      key: 'model',
                                      render: (text) => <Tag>{text}</Tag>,
                                  },
                                  {
                                      title: 'Created At',
                                      dataIndex: 'createdAt',
                                      key: 'createdAt',
                                      render: (text) => new Date(text).toLocaleDateString('en-US', { year: 'numeric', month: '2-digit', day: '2-digit' }),
                                  },
                                  // {
                                  //     title: 'Status',
                                  //     dataIndex: 'isActive',
                                  //     key: 'isActive',
                                  //     render: (text) => (
                                  //         <Tag color={text ? 'green' : 'red'}>
                                  //             {text ? 'Active' : 'Inactive'}
                                  //         </Tag>
                                  //     ),
                                  // },
                                  {
                                      title: 'Actions',
                                      key: 'actions',
                                      render: (text, record) => (
                                          <Space size="middle">
                                              <Button onClick={() => router.push(`/dashboard/models/${record.modelId}`)}>View</Button>
                                          </Space>
                                      ),
                                  },
                              ]}
                              dataSource={data}
                          ></Table>
                        </div>
                      </Card>
                  </Col>
              </Row>
            </DashboardLayout>
        </main>
    </>
  )
}

export default withAuth(checkPagePermissions(['owner', 'admin', 'read', 'write', 'limited'])(ModelsPage));