import checkPagePermissions from '@/components/auth/checkPagePermissions'
import withAuth from '@/components/auth/withAuth'
import DashboardLayout from '@/components/layout/DashboardLayout'
import PageLoading from '@/components/utils/PageLoading'
import AppContext from '@/context/AppContext'
import { Breadcrumb, Card, Col, message, Row, Space, Form, Input, InputNumber, Select, Typography, Alert, Modal, Button, Table, Tag, Popconfirm } from 'antd'
import Head from 'next/head'
import Link from 'next/link'
import { useRouter } from 'next/router'
import React, { useContext, useState, useEffect } from 'react'

const SettingsPage = () => {
  const router = useRouter();
  const context = useContext(AppContext);
  const orgId = context?.activeOrg?._id || null;
  const [messageApi, contextHolder] = message.useMessage();
  const [loading, setLoading] = useState(true);
  const [orgDetails, setOrgDetails] = useState(null);
  const [triggerEffects, setTriggerEffects] = useState(false);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isEditMode, setIsEditMode] = useState(false);
  const [selectedMember, setSelectedMember] = useState(null);
  const [formValues, setFormValues] = useState({
    name: '',
    description: '',
    logoUrl: '',
    members: [],
    isActive: true,
  });
  const [updatedValues, setUpdatedValues] = useState({
    name: '',
    description: '',
    logoUrl: '',
    isActive: true,
  });
  const [formValues2, setFormValues2] = useState({
    email: '',
    access: 'read',
    memberId: '',
  });
  const [form] = Form.useForm();
  const [form2] = Form.useForm();

  const handleEditMember = (member) => {
    setSelectedMember(member);
    setFormValues2({
      email: member.userId.email,
      access: member.access,
      memberId: member.userId._id,
    });
    form2.setFieldsValue({
      email: member.userId.email,
      access: member.access,
    });
    setIsEditMode(true);
    setIsModalOpen(true);
  };

  const handleCancelEdit = () => {
    setIsModalOpen(false);
  };

  useEffect(() => {
    if(!isModalOpen){
      setIsEditMode(false);
      setSelectedMember(null);
      setFormValues2({ email: '', access: 'read', memberId: '' });
      form2.setFieldsValue(
        { email: '', access: 'read' }
      );
    }
  }, [isModalOpen])

  const fetchOrgDetails = async () => {
    if (!orgId) {
      messageApi.error('Organization ID is required to fetch details');
      return;
    }
    try {
      setLoading(true);
      const response = await fetch('/api/organization/detail', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ orgId }),
      });
      const result = await response.json();
      if (result.status === 'success' && result.data) {
        setOrgDetails(result.data);
        setFormValues({
          name: result.data.name,
          description: result.data.description,
          logoUrl: result.data.logoUrl || '',
          members: result.data.members || [],
          isActive: result.data.isActive,
        });
        form.setFieldsValue({
          name: result.data.name,
          description: result.data.description,
          logoUrl: result.data.logoUrl || '',
          isActive: result.data.isActive,
        });
      } else {
        messageApi.error(result.message || 'Failed to fetch organization details');
      }
    } catch (error) {
      console.error('Error fetching organization details:', error);
      messageApi.error('Failed to fetch organization details');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (orgId) {
      fetchOrgDetails();
    }
  }, [orgId, triggerEffects]);

  const isValuesChanged = () => {
    console.log('>>', (formValues.name !== form.getFieldValue('name') ||
           formValues.description !== form.getFieldValue('description') ||
           formValues.logoUrl !== form.getFieldValue('logoUrl')))
    return formValues.name !== form.getFieldValue('name') ||
           formValues.description !== form.getFieldValue('description') ||
           formValues.logoUrl !== form.getFieldValue('logoUrl');
  }
  return (
    <>
      {contextHolder}
      <Head>
        <title>Manage Organization | Sendiee</title>
        <meta name="viewport" content="width=device-width, initial-scale=1" />
        <link rel="icon" href="/favicon.ico" />
      </Head>
      <PageLoading />
      <Modal
        title={isEditMode ? 'Edit Member' : 'Add Member'}
        centered
        open={isModalOpen}
        onOk={async () => {
          setLoading(true);
          try {
            const response = await fetch(isEditMode ? '/api/organization/editmember' : '/api/organization/addmember', {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify({ orgId, email: formValues2.email, access: formValues2.access, orgOwnerId: context?.activeOrg?.ownerId, memberId: formValues2.memberId || '' }),
            });
            const result = await response.json();
            if (result.status === 'success') {
              messageApi.success(result.message || 'Member updated successfully');
              setTriggerEffects(!triggerEffects);
              setIsModalOpen(false);
              setFormValues2({ email: '', access: 'read' });
              // Reset form fields after successful addition
              form2.resetFields();
            } else {
              messageApi.error(result.message || 'Failed to update members list');
            }
          } catch (error) {
            console.error('Error updating members list:', error);
            messageApi.error('Failed to update members list');
          } finally {
            setLoading(false);
          }
        }}
        onCancel={() => handleCancelEdit()}
        okText={isEditMode ? 'Update Member' : 'Add Member'}
        cancelText="Cancel"
        confirmLoading={loading}
      >
        <Form form={form2} layout="vertical" initialValues={formValues2} style={{ marginTop: '10px' }}>
          { !isEditMode && <Typography.Text type="secondary">Enter the email of the user you want to add to your organization.</Typography.Text>}
          <Form.Item
            label=""
            name="email"
            rules={[{ required: true, message: 'Please enter the member email' }]}
            style={{ marginTop: '10px' }}
          >
            <Input size='large' 
              type='email' 
              placeholder="Enter user's email to add" 
              value={formValues2.email} 
              disabled={isEditMode}
              onChange={(e) => setFormValues2({...formValues2, email: e.target.value})} 
            />
          </Form.Item>
          <Form.Item
            label="Access Level"
            name="access"
            rules={[{ required: true, message: 'Please select the access level' }]}
            style={{ marginTop: '10px' }}
          >
            <Select
              size='large'
              placeholder="Select access level"
              value={formValues2.access}
              onChange={(value) => setFormValues2({ ...formValues2, access: value })}
            >
              <Select.Option value="read">Read</Select.Option>
              <Select.Option value="write">Write</Select.Option>
              <Select.Option value="admin">Admin</Select.Option>
              <Select.Option value="limited">Limited</Select.Option>
            </Select>
          </Form.Item>
        </Form>
      </Modal>
      <main>
        <DashboardLayout title="Settings" loading={false} activemenu="settings">
          <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
                <Breadcrumb
                    items={[
                    {
                        title:<Link href={"/dashboard"}>Dashboard</Link>,
                    },
                    {
                        title: 'Settings',
                    },
                    ]}
                />
          </Space>
          <Row gutter={[15, 15]}>
            <Col span={24}>
                <Form
                  form={form}
                  layout="vertical"
                  initialValues={formValues}
                  onValuesChange={(changedValues) => {
                    setUpdatedValues({...updatedValues, ...changedValues });
                  }}
                  onFinish={async (values) => {
                    setLoading(true);
                    try {
                      const response = await fetch('/api/organization/update', {
                        method: 'POST',
                        headers: {
                          'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({ orgId, ...values }),
                      });
                      const result = await response.json();
                      if (result.status === 'success') {
                        messageApi.success(result.message || 'Organization updated successfully');
                        setTriggerEffects(!triggerEffects);
                      } else {
                        messageApi.error(result.message || 'Failed to update organization');
                      }
                    } catch (error) {
                      console.error('Error updating organization:', error);
                      messageApi.error('Failed to update organization');
                    } finally {
                      setLoading(false);
                    }
                  }}
                >
                  <Card title="Organization Settings" extra={
                      <Space>
                        <Button type="primary" htmlType="submit" loading={loading} disabled={!isValuesChanged()}>Update Organization</Button>
                      </Space>
                  }
                  loading={loading}
                  >
                  <Row gutter={[15, 15]}>
                    <Col span={24}>
                      <Form.Item
                        label="Organization Name"
                        name="name"
                        rules={[{ required: true, message: 'Please enter the organization name' }]}
                      >
                        <Input />
                      </Form.Item>
                    </Col>
                    <Col span={24}>
                      <Form.Item
                        label="Description"
                        name="description"
                      >
                        <Input.TextArea rows={4} />
                      </Form.Item>
                    </Col>
                    <Col span={24}>
                    <Card title="Members" extra={
                      <Button type="" onClick={() => setIsModalOpen(true)}>Add Members</Button>
                    }
                    // type="inner"
                    titleStyle={{ fontWeight: 'bold' }}
                    variant="borderless"
                    >
                      <Table
                        dataSource={orgDetails?.members || []}
                        rowKey="userId"
                        size='small'
                        pagination={false}
                        showHeader={false}
                        // bordered={true}
                        columns={[
                          {
                            title: 'Name',
                            dataIndex: ['userId', 'name'],
                            key: 'name',
                          },
                          {
                            title: 'Email',
                            dataIndex: ['userId', 'email'],
                            key: 'email',
                          },
                          {
                            title: 'Access',
                            dataIndex: 'access',
                            key: 'access',
                            render: (text) => <Tag style={{ textTransform: 'capitalize' }}>{text}</Tag>,
                          },
                          {
                            title: 'Actions',
                            key: 'actions',
                            render: (_, record) => (
                              <Space>
                                <Button type="link" onClick={() => {
                                  handleEditMember(record);
                                }}>Edit</Button>
                                <Popconfirm
                                  title="Are you sure you want to remove this member?"
                                  onConfirm={async () => {
                                  try {
                                    const response = await fetch('/api/organization/removemember', {
                                      method: 'POST',
                                      headers: {
                                        'Content-Type': 'application/json',
                                      },
                                      body: JSON.stringify({ orgId, memberId: record.userId }),
                                    });
                                    const result = await response.json();
                                    if (result.status === 'success') {
                                      messageApi.success(result.message || 'Member removed successfully');
                                      setTriggerEffects(!triggerEffects);
                                    } else {
                                      messageApi.error(result.message || 'Failed to remove member');
                                    }
                                  } catch (error) {
                                    console.error('Error removing member:', error);
                                    messageApi.error('Failed to remove member');
                                  }
                                }

                                  }
                                  okText="Yes"
                                  cancelText="No"
                                >
                                  <Button type="link" danger>Remove</Button>
                                </Popconfirm>
                              </Space>
                            ),
                          },
                        ]}
                      ></Table>
                    </Card>
                    </Col>
                  </Row>
                  </Card>
                </Form>
            </Col>
          </Row>
        </DashboardLayout>
      </main>
    </>
  )
}

export default withAuth(checkPagePermissions(['owner', 'admin', 'read', 'write'])(SettingsPage));