import withAuth from '@/components/auth/withAuth'
import DashboardLayout from '@/components/layout/DashboardLayout'
import AppContext from '@/context/AppContext'
import { Card, Form, Input, Button, message, Switch, Row, Col, Space, Tooltip, Select, Modal, Breadcrumb, Flex, Typography, Tag, Alert } from 'antd'
import { PlusOutlined, EditOutlined, DeleteOutlined, CodeOutlined, EyeOutlined } from '@ant-design/icons'
import React, { useState, useContext, useEffect } from 'react'
import { useRouter } from 'next/router'
import Head from 'next/head'

const { Text, Title } = Typography;
import Link from 'next/link'
import checkPagePermissions from '@/components/auth/checkPagePermissions'

const typeOptions = [
  { label: 'String', value: 'string' },
  { label: 'Number', value: 'number' },
  { label: 'Boolean', value: 'boolean' },
  { label: 'Object', value: 'object' },
  { label: 'Array', value: 'array' },
];

function getDefaultProperty(type = 'string') {
  const base = { type, description: '', id: `property_${Math.random().toString(36).slice(2)}` };
  if (type === 'object') return { ...base, properties: {}, required: [] };
  if (type === 'array') return { ...base, items: { type: 'string' } };
  if (type === 'string') return { ...base, enum: [] };
  return base;
}

function buildParameters(properties, requiredList = []) {
  // Recursively build OpenAI-style parameters object
  const props = {};
  Object.entries(properties).forEach(([name, prop]) => {
    const { name: _n, required, ...rest } = prop;
    if (rest.type === 'object') {
      props[name] = {
        ...rest,
        properties: buildParameters(rest.properties || {}, rest.required || []).properties,
        required: rest.required && rest.required.length > 0 ? rest.required : undefined,
      };
    } else if (rest.type === 'array') {
      props[name] = {
        ...rest,
        items: rest.items,
      };
    } else {
      props[name] = rest;
    }
  });
  return {
    type: 'object',
    properties: props,
    required: requiredList.length > 0 ? requiredList : undefined,
  };
}

const PropertyEditor = ({
  visible,
  property,
  onSave,
  onCancel,
  parentRequired,
  isEdit,
}) => {
  const [form] = Form.useForm();
  const [enumValues, setEnumValues] = useState(property.enum || []);
  const [itemType, setItemType] = useState(property.items?.type || 'string');
  const [nestedProps, setNestedProps] = useState(property.properties || {});
  const [nestedRequired, setNestedRequired] = useState(property.required || []);
  const [showEnum, setShowEnum] = useState(property.type === 'string');
  const [showItems, setShowItems] = useState(property.type === 'array');
  const [showNested, setShowNested] = useState(property.type === 'object');
  const [enumInput, setEnumInput] = useState('');

  useEffect(() => {
    form.setFieldsValue({
      ...property,
      enum: undefined,
      items: undefined,
      properties: undefined,
      required: undefined,
    });
    setShowEnum(property.type === 'string');
    setShowItems(property.type === 'array');
    setShowNested(property.type === 'object');
    setEnumValues(property.enum || []);
    setItemType(property.items?.type || 'string');
    setNestedProps(property.properties || {});
    setNestedRequired(property.required || []);
  }, [property, form, visible]);

  const handleTypeChange = (type) => {
    setShowEnum(type === 'string');
    setShowItems(type === 'array');
    setShowNested(type === 'object');
    if (type === 'string') setEnumValues([]);
    if (type === 'array') setItemType('string');
    if (type === 'object') {
      setNestedProps({});
      setNestedRequired([]);
    }
  };

  const handleEnumAdd = (v) => {
    if (v && !enumValues.includes(v)) {
      setEnumValues([...enumValues, v]);
      setEnumInput('');
    }
  };

  const handleEnumRemove = (v) => {
    setEnumValues(enumValues.filter(e => e !== v));
  };

  const handleNestedSave = (name, prop) => {
    setNestedProps({ ...nestedProps, [name]: prop });
  };

  const handleNestedRemove = (name) => {
    const np = { ...nestedProps };
    delete np[name];
    setNestedProps(np);
    setNestedRequired(nestedRequired.filter(r => r !== name));
  };

  const handleNestedRequired = (name, checked) => {
    setNestedRequired(checked
      ? [...new Set([...nestedRequired, name])]
      : nestedRequired.filter(r => r !== name)
    );
  };

  const onFinish = (values) => {
    let prop = {
      ...property,
      ...values,
      type: values.type,
      id: property.id || `property_${Math.random().toString(36).slice(2)}`,
    };
    if (showEnum && enumValues.length > 0) prop.enum = enumValues;
    else delete prop.enum;
    if (showItems) prop.items = { type: itemType };
    else delete prop.items;
    if (showNested) {
      prop.properties = nestedProps;
      if (nestedRequired.length > 0) prop.required = nestedRequired;
      else delete prop.required;
    } else {
      delete prop.properties;
      delete prop.required;
    }
    onSave(prop);
  };

  return (
    <Modal
      open={visible}
      title={isEdit ? 'Edit Property' : 'Add Property'}
      onCancel={onCancel}
      onOk={() => form.submit()}
      width={showNested ? 700 : 500}
      destroyOnClose
    >
      <Form
        form={form}
        layout="vertical"
        initialValues={property}
        onFinish={onFinish}
      >
        <Form.Item
          name="name"
          label="Property Name"
          rules={[{ required: true, message: 'Enter property name' },
            {
              validator: (_, value) => {
                if (!value || /^[a-z_]+$/.test(value)) {
                  return Promise.resolve();
                }
                return Promise.reject(
                  new Error('Tool name must contain only lowercase letters and underscores (no spaces or uppercase letters)')
                );
              },
            },
          ]}
        >
          <Input />
        </Form.Item>
        <Form.Item
          name="type"
          label="Type"
          rules={[{ required: true, message: 'Select type' }]}
        >
          <Select options={typeOptions} onChange={handleTypeChange} />
        </Form.Item>
        <Form.Item name="description" label="Description">
          <Input.TextArea rows={2} />
        </Form.Item>
        {/* <Form.Item name="required" label="Required" valuePropName="checked">
          <Switch checked={parentRequired}  />
        </Form.Item> */}
        {showEnum && (
          <Form.Item label="Enum Values">
            <Space direction="vertical" style={{ width: '100%' }}>
              <Input.Search
                placeholder="Add enum value"
                enterButton="Add"
                value={enumInput}
                onChange={(e) => setEnumInput(e.target.value)}
                onSearch={handleEnumAdd}
              />
              <Space wrap>
                {enumValues.map(v => (
                  <Tag key={v} style={{ padding: '2px 8px', borderRadius: 4, marginRight: 4 }}>
                    {v}
                    <DeleteOutlined style={{ marginLeft: 6, color: '#f5222d', cursor: 'pointer' }} onClick={() => handleEnumRemove(v)} />
                  </Tag>
                ))}
              </Space>
            </Space>
          </Form.Item>
        )}
        {showItems && (
          <Form.Item label="Array Item Type">
            <Select value={itemType} onChange={setItemType} options={typeOptions.filter(t => t.value !== 'array')} />
          </Form.Item>
        )}
        {showNested && (
          <Form.Item label="Nested Properties">
            <PropertyList
              properties={nestedProps}
              requiredList={nestedRequired}
              onSave={handleNestedSave}
              onRemove={handleNestedRemove}
              onRequired={handleNestedRequired}
              isNested
            />
            <div style={{ marginTop: 8 }}>
              <Button type="dashed" icon={<PlusOutlined />} onClick={() => handleNestedSave(`prop_${Date.now()}`, getDefaultProperty())}>
                Add Nested Property
              </Button>
            </div>
          </Form.Item>
        )}
      </Form>
    </Modal>
  );
};

const PropertyList = ({
  properties,
  requiredList,
  onSave,
  onRemove,
  onRequired,
  isNested = false,
}) => {
  const [editing, setEditing] = useState(null);
  const [editProp, setEditProp] = useState(null);

  const handleEdit = (name, prop) => {
    setEditing(name);
    setEditProp({ ...prop, name });
  };

  const handleSave = (prop) => {
    onSave(prop.name, prop);
    setEditing(null);
    setEditProp(null);
  };

  return (
    <div>
      {Object.entries(properties).map(([name, prop]) => (
        <Card size='small' style={{ marginBottom: 8, borderStyle: 'dashed' }} key={prop.id || name}>
            <Row key={prop.id || name} gutter={8} align="middle" style={{ marginBottom: 6 }}>
            <Col flex="auto">
                <b>{name}</b> <span style={{ color: '#888' }}>({prop.type})</span>
                {prop.description && <Tooltip title={prop.description}><span style={{ marginLeft: 8, color: '#aaa' }}>{prop.description}</span></Tooltip>}
            </Col>
            <Col>
            <Space>
                <Tooltip title="Required">
                    <Switch
                        checked={requiredList?.includes(name)}
                        onChange={checked => onRequired(name, checked)}
                        size="small"
                        style={{ marginRight: 8 }}
                    />
                </Tooltip>
                <Tooltip title="Edit">
                <Button type='text' icon={<EditOutlined />} size="small" onClick={() => handleEdit(name, prop)} />
                </Tooltip>
                <Tooltip title="Remove">
                <Button type='link' icon={<DeleteOutlined />} size="small" danger onClick={() => onRemove(name)} />
                </Tooltip>
            </Space>
            </Col>
            </Row>
        </Card>
      ))}
      {editing && (
        <PropertyEditor
          visible={!!editing}
          property={editProp}
          onSave={handleSave}
          onCancel={() => setEditing(null)}
          parentRequired={requiredList?.includes(editProp?.name)}
          isEdit
        />
      )}
    </div>
  );
};

const ParametersEditor = ({ value, onChange }) => {
  const [properties, setProperties] = useState(value?.properties || {});
  const [requiredList, setRequiredList] = useState(value?.required || []);
  const [editing, setEditing] = useState(false);
  const [editProp, setEditProp] = useState(null);
  const [visual, setVisual] = useState(true);
  const [jsonCode, setJsonCode] = useState(JSON.stringify(value || { type: 'object', properties: {}, required: [] }, null, 2));
  const [jsonError, setJsonError] = useState('');

  useEffect(() => {
    setProperties(value?.properties || {});
    setRequiredList(value?.required || []);
    setJsonCode(JSON.stringify(value || { type: 'object', properties: {}, required: [] }, null, 2));
  }, [value]);

  const handleAdd = () => {
    setEditProp(getDefaultProperty());
    setEditing(true);
  };

  const handleSave = (prop) => {
    const name = prop.name;
    const newProps = { ...properties, [name]: prop };
    const newRequired = prop.required ? [...new Set([...requiredList, name])] : requiredList.filter(r => r !== name);
    setProperties(newProps);
    setRequiredList(newRequired);
    setEditing(false);
    setEditProp(null);
    const params = buildParameters(newProps, newRequired);
    onChange(params);
  };

  const handleRemove = (name) => {
    const newProps = { ...properties };
    delete newProps[name];
    const newRequired = requiredList.filter(r => r !== name);
    setProperties(newProps);
    setRequiredList(newRequired);
    const params = buildParameters(newProps, newRequired);
    onChange(params);
  };

  const handleRequired = (name, checked) => {
    const newRequired = checked
      ? [...new Set([...requiredList, name])]
      : requiredList.filter(r => r !== name);
    setRequiredList(newRequired);
    const params = buildParameters(properties, newRequired);
    onChange(params);
  };

  const handleVisualToggle = (checked) => {
    setVisual(checked);
    if (!checked) {
      setJsonCode(JSON.stringify(buildParameters(properties, requiredList), null, 2));
    }
  };

  const handleJsonChange = (e) => {
    setJsonCode(e.target.value);
    try {
      const parsed = JSON.parse(e.target.value);
      setJsonError('');
      setProperties(parsed.properties || {});
      setRequiredList(parsed.required || []);
      onChange(parsed);
    } catch (err) {
      setJsonError('Invalid JSON');
    }
  };

  return (
    <div>
      <Row align="middle" style={{ marginBottom: 8 }}>
        <Col span={24}>
        <Card title={<Title style={{ margin: 0 }} level={4}>Parameters</Title>} style={{ marginBottom: 16 }} extra={
            <Flex align='center'>
                <Text style={{ marginRight: 8 }}>{visual ? 'Visual Editor' : 'JSON Editor'}</Text>
                <Switch
                checked={visual}
                onChange={handleVisualToggle}
                checkedChildren={<EyeOutlined />}
                unCheckedChildren={<CodeOutlined />}
              />
            </Flex>
        }>
        {visual ? (
            <>
            <PropertyList
                properties={properties}
                requiredList={requiredList}
                onSave={(name, prop) => {
                const newProps = { ...properties, [name]: prop };
                setProperties(newProps);
                const newRequired = prop.required ? [...new Set([...requiredList, name])] : requiredList.filter(r => r !== name);
                setRequiredList(newRequired);
                const params = buildParameters(newProps, newRequired);
                onChange(params);
                }}
                onRemove={handleRemove}
                onRequired={handleRequired}
            />
            <Button type="dashed" block icon={<PlusOutlined />} onClick={handleAdd} style={{ marginTop: 8 }}>
                Add Property
            </Button>
            {editing && (
                <PropertyEditor
                visible={editing}
                property={editProp}
                onSave={handleSave}
                onCancel={() => setEditing(false)}
                parentRequired={editProp?.required}
                isEdit={false}
                />
            )}
            </>
        ) : (
            <>
            <Input.TextArea
                rows={10}
                value={jsonCode}
                onChange={handleJsonChange}
                style={{ fontFamily: 'monospace' }}
            />
            {jsonError && <div style={{ color: 'red' }}>{jsonError}</div>}
            </>
        )}
        </Card>
        </Col>
      </Row>
    </div>
  );
};

const CreateToolPage = () => {
  const [loading, setLoading] = useState(false);
  const router = useRouter();
  const context = useContext(AppContext);
  const [messageApi, contextHolder] = message.useMessage();
  const [form] = Form.useForm();
  const [parameters, setParameters] = useState({ type: 'object', properties: {}, required: [] });
  const [formValues, setFormValues] = useState({});

  const handleSubmit = async (values) => {
    setLoading(true);
    try {
      const res = await fetch('/api/tools/create', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          ...values,
          orgId: context?.activeOrg?._id,
          toolType: 'function',
          parameters,
        }),
      });
      const data = await res.json();
      if (data.status === 'success') {
        messageApi.success('Tool created successfully!');
        router.push('/dashboard/tools');
      } else {
        messageApi.error(data.message || 'Failed to create tool');
      }
    } catch (err) {
      messageApi.error('Error creating tool');
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
    <Head>
      <title>Create Tool | Sendiee</title>
    </Head>
    <DashboardLayout title='Create Tool' activemenu='tools'>
      {contextHolder}
        <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
            <Breadcrumb
                items={[
                {
                    title:<Link href={"/dashboard"}>Dashboard</Link>,
                },
                {
                    title:<Link href={"/dashboard/tools"}>Tools</Link>,
                },
                {
                    title: 'Create Tool',
                },
                ]}
            />
        </Space>
        <Card title="Create Tool" style={{ maxWidth: 700, margin: '0 auto' }}>
            <Form
                form={form}
                layout="vertical"
                onFinish={handleSubmit}
                onValuesChange={() => {setFormValues({...formValues, ...form.getFieldsValue()})}}
                initialValues={{
                    isActive: true,
                }}
            >
                <Row gutter={[10, 10]}>
                    <Col span={24}>
                        <Form.Item
                            name="name"
                            label="Tool Name"
                            rules={[{ required: true, message: 'Please enter tool name' },
                              {
                                validator: (_, value) => {
                                  if (!value || /^[a-z_]+$/.test(value)) {
                                    return Promise.resolve();
                                  }
                                  return Promise.reject(
                                    new Error('Tool name must contain only lowercase letters and underscores (no spaces or uppercase letters)')
                                  );
                                },
                              },
                            ]}
                        >
                            <Input placeholder="Enter tool name" size='large' />
                        </Form.Item>
                    </Col>
                    <Col span={24}>
                        <Form.Item
                            name="description"
                            label="Description (When to use this tool)"
                            rules={[{ required: true, message: 'Please enter tool description' }]}
                        >
                            <Input.TextArea rows={3} placeholder="Describe this tool" size='large' />
                        </Form.Item>
                    </Col>
                    <Col span={24}>
                    <Card size='small' style={{ marginBottom: 16 }}>  
                        <Flex justify="space-between" align="center">
                            <Flex vertical>
                                <Title level={4}>Static</Title>
                                <Text>
                                    Tool will return a static response instead of executing a function
                                </Text>
                            </Flex>
                            <Form.Item
                                name="isStatic"
                                valuePropName="checked"
                            >
                                <Switch />
                            </Form.Item>
                        </Flex>
                    </Card>
                    </Col>
                    {
                        (formValues.isStatic === true) ? (
                            <>
                            <Col span={24}>
                                <Form.Item
                                    name="response"
                                    label="Static Response"
                                    rules={[{ required: true, message: 'Please enter response message' }]}
                                >
                                    <Input.TextArea rows={3} placeholder="Response message to be sent to user" size='large' />
                                </Form.Item>
                            </Col>
                            <Col span={24}>
                              <Alert 
                                description="Below fields are not required for static tool."
                                type="info"
                                showIcon
                              />
                            </Col>
                            </>
                        ) : null
                    }
                    <Col span={24}>
                        <Form.Item
                            name="serverUrl"
                            label="Server URL"
                            // rules={[{ required: true, message: 'Please enter server URL' }]}
                        >
                            <Input placeholder="https://your-server.com/endpoint" size='large' />
                        </Form.Item>
                    </Col>
                    <Col span={24}>
                        <Form.Item
                            name="secretToken"
                            label="Secret Token"
                        >
                            <Input.Password placeholder="Secret token (optional)" size='large' />
                        </Form.Item>
                    </Col>
                    <Col span={24}>
                    <Card size='small'>  
                        <Flex justify="space-between" align="center">
                            <Flex vertical>
                                <Title level={4}>Strict</Title>
                                <Text>Enforces strict parameter validation</Text>
                            </Flex>
                            <Form.Item
                                name="strict"
                                valuePropName="checked"
                            >
                                <Switch />
                            </Form.Item>
                        </Flex>
                    </Card>
                    </Col>
                    <Col span={24}>
                    <Card size='small'>  
                        <Flex justify="space-between" align="center">
                            <Flex vertical>
                                <Title level={4}>Async</Title>
                                <Text>Tool executes asynchronously</Text>
                            </Flex>
                            <Form.Item
                                name="async"
                                valuePropName="checked"
                            >
                                <Switch />
                            </Form.Item>
                        </Flex>
                    </Card>
                    </Col>
                    <Col span={24}>
                        <Form.Item
                            required
                        >
                            <ParametersEditor value={parameters} onChange={setParameters} />
                        </Form.Item>
                    </Col>
                    <Col span={24}>
                        <Form.Item>
                            <Button type="primary" htmlType="submit" loading={loading} block>
                            Create Tool
                            </Button>
                        </Form.Item>
                    </Col>
                </Row>
                {/* <Form.Item
                    name="isActive"
                    label="Active"
                    valuePropName="checked"
                >
                    <Switch defaultChecked />
                </Form.Item> */}
            </Form>
        </Card>
    </DashboardLayout>
    </>
  );
};

export default withAuth(checkPagePermissions(['owner', 'admin', 'write'])(CreateToolPage));