import checkPagePermissions from '@/components/auth/checkPagePermissions'
import withAuth from '@/components/auth/withAuth'
import DashboardLayout from '@/components/layout/DashboardLayout'
import PageLoading from '@/components/utils/PageLoading'
import AppContext from '@/context/AppContext'
import { Breadcrumb, Button, Card, Col, message, Popconfirm, Row, Space, Table, Tag, Switch } from 'antd'
import Head from 'next/head'
import Link from 'next/link'
import { useRouter } from 'next/router'
import React, { useContext, useEffect, useState } from 'react'

const ToolsPage = () => {
  const [loading, setLoading] = useState(false);
  const [data, setData] = useState([]);
  const [triggerEffects, setTriggerEffects] = useState(false);
  const [messageApi, contextHolder] = message.useMessage();
  const context = useContext(AppContext);
  const router = useRouter();
  const orgId = context?.activeOrg?._id || null;

  const fetchModels = async () => {
    setLoading(true);
    try {
      const response = await fetch('/api/tools/list', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ orgId }),
      });
      const result = await response.json();
      if (result.status === 'success') {
        setData(result.data);
      } else {
        messageApi.error(result.message || 'Failed to fetch AI tools');
      }
    } catch (error) {
      console.error('Error fetching AI tools:', error);
      messageApi.error('Failed to fetch AI tools');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (orgId) {
      fetchModels();
    } else {
      messageApi.error('Organization ID is required to fetch tools');
    }
  }, [orgId, triggerEffects]);

  const handleToggleToolStatus = async (toolId, isActive, _id) => {
    setLoading(true);
    try {
      const response = await fetch('/api/tools/toggle', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ toolId, isActive, orgId, _id }),
      });
      const result = await response.json();
      if (result.status === 'success') {
        messageApi.success(result.message || 'Tool status updated successfully');
        setTriggerEffects(!triggerEffects);
      } else {
        messageApi.error(result.message || 'Failed to update tool status');
      } 
    } catch (error) {
      console.error('Error updating tool status:', error);
      messageApi.error('Failed to update tool status');
    } finally {
      setLoading(false);
    }
  }

  return (
    <>
      {contextHolder}
        <Head>
            <title>Tools | Sendiee Dashboard</title>
            <meta name="description" content="Sendiee dashboard" />
            <meta name="viewport" content="width=device-width, initial-scale=1" />
            <link rel="icon" href="/favicon.ico" />
        </Head>
        <PageLoading />
        <main>
            <DashboardLayout title="Tools" loading={false} activemenu="tools">
              <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
                    <Breadcrumb
                        items={[
                        {
                            title:<Link href={"/dashboard"}>Dashboard</Link>,
                        },
                        {
                            title: 'Tools',
                        },
                        ]}
                    />
              </Space>
              <Row gutter={[15, 15]}>
                  <Col span={24}>
                      <Card title="AI Tools" extra={
                          <Space>
                              <Button type='primary' onClick={() => {router.push('/dashboard/tools/create')}}>Add Tool</Button>
                          </Space>
                      }>
                        <div className='table_container'>
                          <Table 
                              bordered={true}
                              loading={loading}
                              rowKey="toolId"
                              size='small'
                              columns={[
                                  {
                                      title: 'Status',
                                      dataIndex: 'isActive',
                                      key: 'isActive',
                                      render: (text, record) => (
                                          <Switch 
                                              checked={text}
                                              onChange={(checked) => handleToggleToolStatus(record.toolId, checked, record._id )}
                                              loading={loading}
                                          />
                                      ),
                                  },
                                  {
                                      title: 'Tool Name',
                                      dataIndex: 'name',
                                      key: 'name',
                                  },
                                  {
                                      title: 'Server',
                                      dataIndex: 'serverUrl',
                                      key: 'serverUrl',
                                      render: (text) => <Tag>{text}</Tag>,
                                  },
                                  {
                                      title: 'Created At',
                                      dataIndex: 'createdAt',
                                      key: 'createdAt',
                                      render: (text) => new Date(text).toLocaleDateString('en-US', { year: 'numeric', month: '2-digit', day: '2-digit' }),
                                  },
                                  {
                                      title: 'Actions',
                                      key: 'actions',
                                      render: (text, record) => (
                                          <Space size="middle">
                                              <Button onClick={() => router.push(`/dashboard/tools/${record.toolId}`)}>View</Button>
                                          </Space>
                                      ),
                                  },
                              ]}
                              dataSource={data}
                          ></Table>
                        </div>
                      </Card>
                  </Col>
              </Row>
            </DashboardLayout>
        </main>
    </>
  )
}

export default withAuth(checkPagePermissions(['owner', 'admin', 'read', 'write', 'limited'])(ToolsPage));