import checkPagePermissions from '@/components/auth/checkPagePermissions'
import withAuth from '@/components/auth/withAuth'
import DashboardLayout from '@/components/layout/DashboardLayout'
import OnboardCustomer from '@/components/meta/OnboardCustomer'
import PageLoading from '@/components/utils/PageLoading'
import AppContext from '@/context/AppContext'
import LinkWhatsAppModal from '@/modals/LinkWhatsAppModal'
import { Breadcrumb, Button, Card, Col, message, Modal, Popconfirm, QRCode, Row, Space, Table, Tag } from 'antd'
import Head from 'next/head'
import Link from 'next/link'
import React, { useContext, useEffect, useState } from 'react'

const WAAccountsPage = () => {
    const [loading, setLoading] = useState(false);
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [messageApi, contextHolder] = message.useMessage();
    const [data, setData] = useState([]);
    const [triggerEffects, setTriggerEffects] = useState(false);
    const [selectedInstance, setSelectedInstance] = useState(null);
    const context = useContext(AppContext);

    useEffect(() => {
        const fetchData = async () => {
            setLoading(true);
            setData([]);
            try {
                const response = await fetch('/api/instance/list', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        orgId: context?.activeOrg?._id,
                    }),
                });
                const result = await response.json();
                if (result.status === 'success') {
                    setData(result.data);
                } else {
                    messageApi.open({
                        type: 'error',
                        content: result.message || 'Failed to fetch WhatsApp accounts',
                    });
                }
            } catch (error) {
                messageApi.open({
                    type: 'error',
                    content: error.message || 'An error occurred while fetching WhatsApp accounts',
                });
            } finally {
                setLoading(false);
            }
        };
        fetchData();
    }, [triggerEffects]);

    const handleViewQR = async (instance) => {
        setSelectedInstance(instance);
        setIsModalOpen(true);
    }

  return (
    <>
        <Head>
            <title>WhatsApp Accounts | Sendiee</title>
            <meta name="viewport" content="width=device-width, initial-scale=1" />
        </Head>
        {contextHolder}
        <PageLoading />
        <LinkWhatsAppModal isOpen={isModalOpen} onClose={setIsModalOpen} onLinkWhatsApp={() => {setTriggerEffects(!triggerEffects)}} selectedInstance={selectedInstance} setSelectedInstance={(data) => setSelectedInstance(data)} />
        <main>
            <DashboardLayout title="WhatsApp Accounts" loading={false} activemenu="waaccounts">
                <Space className='space_between' align='center' style={{marginBottom: '15px'}}>
                    <Breadcrumb
                        items={[
                        {
                            title:<Link href={"/dashboard"}>Dashboard</Link>,
                        },
                        {
                            title: 'WhatsApp Accounts',
                        },
                        ]}
                    />
                </Space>
                <Row gutter={[15, 15]}>
                    <Col span={24}>
                        <Card title="WhatsApp Accounts" extra={
                            <Space>
                                {/* <OnboardCustomer /> */}
                                <Button type='primary' onClick={() => {setSelectedInstance(null); setIsModalOpen(true)}}>Link WhatsApp</Button>
                            </Space>
                        }>
                            <div className='table_container'>
                                <Table 
                                    bordered={true}
                                    loading={loading}
                                    columns={[
                                        // {
                                        //     title: 'Instance ID',
                                        //     dataIndex: 'instance_id',
                                        //     key: 'instance_id',
                                        // },
                                        {
                                            title: 'Name',
                                            dataIndex: 'name',
                                            key: 'name',
                                        },
                                        {
                                            title: 'Phone Number',
                                            dataIndex: 'phone',
                                            key: 'phone',
                                        },
                                        {
                                            title: 'Status',
                                            dataIndex: 'status',
                                            key: 'status',
                                            render: (text) => (
                                                <Tag color={text === 'scanned' ? 'green' : (text === 'loading' ? 'blue' : (text === 'expired' ? 'red' : 'orange'))}>
                                                    {(text === 'scanned' && 'Linked') || (text === 'loading' && 'Loading') || (text === 'expired' && 'Expired') || (text === 'active' && 'Active')}
                                                </Tag>
                                            ),
                                        },
                                        {
                                            title: 'Actions',
                                            key: 'actions',
                                            render: (text, record) => (
                                                <Space size="middle">
                                                    { record.status !== 'scanned' && record.status !== 'expired' && <Button onClick={() => handleViewQR(record)} type='link'>View QR</Button>}
                                                    <Popconfirm
                                                        title="Are you sure you want to unlink this WhatsApp account?"
                                                        onConfirm={async () => {
                                                            setLoading(true);
                                                            try {
                                                                const response = await fetch('/api/instance/delete', {
                                                                    method: 'POST',
                                                                    headers: {
                                                                        'Content-Type': 'application/json',
                                                                    },
                                                                    body: JSON.stringify({
                                                                        instance_id: record.instance_id,
                                                                        serverUrl: record.serverUrl,
                                                                        orgId: context?.activeOrg?._id,
                                                                    }),
                                                                });
                                                                const result = await response.json();
                                                                if (result.status === 'success') {
                                                                    messageApi.open({
                                                                        type: 'success',
                                                                        content: result.message || 'WhatsApp account unlinked successfully',
                                                                    });
                                                                    setTriggerEffects(!triggerEffects);
                                                                } else {
                                                                    messageApi.open({
                                                                        type: 'error',
                                                                        content: result.message || 'Failed to unlink WhatsApp account',
                                                                    });
                                                                }
                                                            } catch (error) {
                                                                messageApi.open({
                                                                    type: 'error',
                                                                    content: error.message || 'An error occurred while unlinking WhatsApp account',
                                                                });
                                                            } finally {
                                                                setLoading(false);
                                                            }
                                                        }}
                                                        okText="Yes, unlink it"
                                                        okButtonProps={{ danger: true }}
                                                        cancelText="No"
                                                    >
                                                    <Button type='link' danger>Unlink</Button>
                                                    </Popconfirm>
                                                </Space>
                                            ),
                                        },
                                    ]}
                                    dataSource={data}
                                ></Table>
                            </div>
                        </Card>
                    </Col>
                </Row>
            </DashboardLayout>
        </main>
    </>
  )
}

export default withAuth(checkPagePermissions(['owner', 'admin', 'read', 'write', 'limited'])(WAAccountsPage));